/*
 * The contents of this file are subject to the MonetDB Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.monetdb.org/Legal/MonetDBLicense
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is the MonetDB Database System.
 *
 * The Initial Developer of the Original Code is CWI.
 * Portions created by CWI are Copyright (C) 1997-July 2008 CWI.
 * Copyright August 2008-2015 MonetDB B.V.
 * All Rights Reserved.
 */

/*
 * (author) M.L.Kersten
 * Every MAL command introduced in an atom module should be checked
 * to detect overloading of a predefined function.
 * Subsequently, we update the GDK atom structure.
 * The function signatures should be parameter-less, which
 * enables additional functions with the same name to appear
 * as ordinary mal operators.
 *
 * A few fields are set only once, at creation time.
 * They should be implemented with parameter-less functions.
 */
#include "monetdb_config.h"
#include "mal_instruction.h"
#include "mal_atom.h"
#include "mal_namespace.h"
#include "mal_exception.h"
#include "mal_private.h"

#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

static void setAtomName(InstrPtr pci)
{
	char buf[MAXPATHLEN];
	snprintf(buf, MAXPATHLEN, "#%s", getFunctionId(pci));
	setFunctionId(pci, putName(buf, strlen(buf)));
}

int malAtomProperty(MalBlkPtr mb, InstrPtr pci)
{
	str name;
	int tpe;
	(void)mb;  /* fool compilers */
	assert(pci != 0);
	name = getFunctionId(pci);
	tpe = getTypeIndex(getModuleId(pci), (int)strlen(getModuleId(pci)), TYPE_any);
	if (tpe < 0 || tpe >= GDKatomcnt || tpe >= MAXATOMS)
		return 0;
	assert(pci->fcn != NULL);
	switch (name[0]) {
	case 'd':
		if (idcmp("del", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomDel = (void (*)(Heap *, var_t *))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		break;
	case 'c':
		if (idcmp("cmp", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomCmp = (int (*)(const void *, const void *))pci->fcn;
			BATatoms[tpe].linear = 1;
			setAtomName(pci);
			return 1;
		}
		break;
	case 'f':
		if (idcmp("fromstr", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomFromStr = (int (*)(const char *, int *, ptr *))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		if (idcmp("fix", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomFix = (int (*)(const void *))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		break;
	case 'h':
		if (idcmp("heap", name) == 0 && pci->argc == 1) {
			/* heap function makes an atom varsized */
			BATatoms[tpe].size = sizeof(var_t);
			assert_shift_width(ATOMelmshift(BATatoms[tpe].size), BATatoms[tpe].size);
			BATatoms[tpe].align = sizeof(var_t);
			BATatoms[tpe].atomHeap = (void (*)(Heap *, size_t))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		if (idcmp("hash", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomHash = (BUN (*)(const void *))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		break;
	case 'l':
		if (idcmp("length", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomLen = (int (*)(const void *))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		break;
	case 'n':
		if (idcmp("null", name) == 0 && pci->argc == 1) {
			ptr atmnull = ((ptr (*)(void))pci->fcn)();

			BATatoms[tpe].atomNull = atmnull;
			setAtomName(pci);
			return 1;
		}
		if (idcmp("nequal", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomCmp = (int (*)(const void *, const void *))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		break;
	case 'p':
		if (idcmp("put", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomPut = (var_t (*)(Heap *, var_t *, const void *))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		break;
	case 's':
		if (idcmp("storage", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].storage = (*(long (*)(void))pci->fcn)();
			setAtomName(pci);
			return 1;
		}
		break;
	case 't':
		if (idcmp("tostr", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomToStr = (int (*)(str *, int *, const void *))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		break;
	case 'u':
		if (idcmp("unfix", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomUnfix = (int (*)(const void *))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		break;
	case 'r':
		if (idcmp("read", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomRead = (void *(*)(void *, stream *, size_t))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		break;
	case 'w':
		if (idcmp("write", name) == 0 && pci->argc == 1) {
			BATatoms[tpe].atomWrite = (int (*)(const void *, stream *, size_t))pci->fcn;
			setAtomName(pci);
			return 1;
		}
		break;
	}
	return 0;
}
/*
 * Atoms are constructed incrementally in the kernel using the
 * ATOMallocate function. It takes an existing type as a base
 * to derive a new one.
 * The most tedisous work is to check the signature types of the functions
 * acceptable for the kernel.
 */

void malAtomDefinition(stream *out, str name, int tpe)
{
	int i;

	if (strlen(name) >= IDLENGTH) {
		showException(out, SYNTAX, "atomDefinition", "Atom name '%s' too long", name);
		return;
	}
	if (ATOMindex(name) >= 0) {
		showException(out, TYPE, "atomDefinition", "Redefinition of atom '%s'", name);
		return;
	}
	if (tpe < 0 || tpe >= GDKatomcnt) {
		showException(out, TYPE, "atomDefinition", "Undefined atom inheritance '%s'", name);
		return;
	}

	if (strlen(name) >= sizeof(BATatoms[0].name))
		return;
	i = ATOMallocate(name);
	/* overload atom ? */
	if (tpe) {
		BATatoms[i] = BATatoms[tpe];
		strncpy(BATatoms[i].name, name, sizeof(BATatoms[i].name));
		BATatoms[i].name[sizeof(BATatoms[i].name) - 1] = 0; /* make coverity happy */
		BATatoms[i].storage = BATatoms[tpe].storage;
	} else { /* cannot overload void atoms */
		BATatoms[i].storage = i;
		BATatoms[i].linear = 0;
	}
}
/*
 * User defined modules may introduce fixed sized types
 * to store information in BATs.
 */
int malAtomSize(int size, int align, char *name)
{
	int i = 0;

	i = ATOMindex(name);
	BATatoms[i].storage = i;
	BATatoms[i].size = size;
	assert_shift_width(ATOMelmshift(BATatoms[i].size), BATatoms[i].size);
	BATatoms[i].align = align;
	return i;
}

void showAtoms(stream *fd)
{
	int i;
	for (i = 0; i< GDKatomcnt && BATatoms[i].name[0]; i++) {
		mnstr_printf(fd, "%s", BATatoms[i].name);
		if (BATatoms[i + 1].name[0]) mnstr_printf(fd, ",");
	}
	mnstr_printf(fd, "\n");
}
