/*
 * The contents of this file are subject to the MonetDB Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.monetdb.org/Legal/MonetDBLicense
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is the MonetDB Database System.
 *
 * The Initial Developer of the Original Code is CWI.
 * Portions created by CWI are Copyright (C) 1997-July 2008 CWI.
 * Copyright August 2008-2015 MonetDB B.V.
 * All Rights Reserved.
 */

package nl.cwi.monetdb.jdbc;

import java.sql.*;
import java.io.*;
import java.util.*;
import java.math.*;
import java.net.*;
import java.text.*;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import nl.cwi.monetdb.mcl.parser.*;

/**
 * A ResultSet suitable for the MonetDB database.
 * <br /><br />
 * A table of data representing a database result set, which is usually
 * generated by executing a statement that queries the database.
 * <br /><br />
 * A ResultSet object maintains a cursor pointing to its current row of data.
 * Initially the cursor is positioned before the first row. The next method
 * moves the cursor to the next row, and because it returns false when there
 * are no more rows in the ResultSet object, it can be used in a while loop to
 * iterate through the result set.
 * <br /><br />
 * The current state of this ResultSet is that it supports positioning in the
 * result set, absolute and relative. A slight performance difference between
 * FORWARD_ONLY or result sets scrollable in both directions can be noticed as
 * for FORWARD_ONLY result sets the memory usage will be likely lower for large
 * result sets.
 *
 * @author Fabian Groffen <Fabian.Groffen@cwi.nl>
 * @version 0.7
 */
public class MonetResultSet extends MonetWrapper implements ResultSet {
	/** The last column read using some getXXX function */
	private int lastColumnRead = -1;
	// the following have default access modifier for the MonetVirtualResultSet
	/** The current line of the buffer split in columns */
	final TupleLineParser tlp;
	/** The current position of the cursor for this ResultSet object */
	int curRow = 0;

	// a blank final is immutable once assigned in the constructor
	/** A Header to retrieve lines from */
	private final MonetConnection.ResultSetResponse header;
	/** The names of the columns in this ResultSet */
	private final String[] columns;
	/** The MonetDB types of the columns in this ResultSet */
	private final String[] types;
	/** The id of this ResultSet (needed for closing) */
	private final String tableID;
	/** The number of rows in this ResultSet */
	final int tupleCount;	// default for the MonetVirtualResultSet

	/** The parental Statement object */
	private final Statement statement;

	/** The type of this ResultSet (forward or scrollable) */
	private int type = TYPE_FORWARD_ONLY;
	/** The concurrency of this ResultSet (currently only read-only) */
	private int concurrency = CONCUR_READ_ONLY;
	/** The warnings for this ResultSet object */
	private SQLWarning warnings;

	/**
	 * Main constructor backed by the given Header.
	 *
	 * @param statement the statement which created this ResultSet
	 * @param header a header containing the query, resultset type, etc.
	 * @throws SQLException is a protocol error occurs
	 */
	MonetResultSet(
		Statement statement,
		MonetConnection.ResultSetResponse header)
		throws SQLException
	{
		this.statement = statement;
		this.header = header;
		this.type = header.getRSType();
		this.concurrency = header.getRSConcur();
		// well there is only one supported concurrency, so we don't have to
		// bother about that

		// throws SQLException on getters of Header, so we find out immediately
		// if an error occurred for this query
		columns = header.getNames();
		types = header.getTypes();
		tableID = "" + header.id;
		tupleCount = header.tuplecount;

		// create result array
		tlp = new TupleLineParser(columns.length);
	}

	/**
	 * Constructor used by MonetFillableResultSet.
	 * DO NOT USE THIS CONSTRUCTOR IF YOU ARE NOT EXTENDING THIS OBJECT!
	 *
	 * @param columns the column names
	 * @param types the column types
	 * @param results the number of rows in the ResultSet
	 * @throws IOException if communicating with monet failed
	 * @throws SQLException is a protocol error occurs
	 */
	MonetResultSet(
		String[] columns,
		String[] types,
		int results
	) throws IllegalArgumentException
	{
		if (columns == null || types == null) {
			throw new IllegalArgumentException("One of the given arguments is null!");
		}
		if (columns.length != types.length) {
			throw new IllegalArgumentException("Given arguments are not the same size!");
		}
		if (results < 0) {
			throw new IllegalArgumentException("Negative rowcount not allowed!");
		}

		this.header = null;
		this.tableID = null;
		this.statement = null; // no parent, required for specs

		this.columns = columns;
		this.types = types;
		this.tupleCount = results;

		this.tlp = new TupleLineParser(columns.length);
	}

	//== methods of interface ResultSet

	// Chapter 14.2.2 Sun JDBC 3.0 Specification
	/**
	 * Moves the cursor to the given row number in this ResultSet object.
	 * <br /><br />
	 * If the row number is positive, the cursor moves to the given row number
	 * with respect to the beginning of the result set. The first row is row 1,
	 * the second is row 2, and so on.
	 * <br /><br />
	 * If the given row number is negative, the cursor moves to an absolute row
	 * position with respect to the end of the result set. For example, calling
	 * the method absolute(-1) positions the cursor on the last row; calling the
	 * method absolute(-2) moves the cursor to the next-to-last row, and so on.
	 * <br /><br />
	 * An attempt to position the cursor beyond the first/last row in the result
	 * set leaves the cursor before the first row or after the last row.
	 * Note: calling absolute(1) is the same as calling first(). Calling
	 *       absolute(-1) is the same as calling last().
	 *
	 * @param row the number of the row to which the cursor should move. A
	 *        positive number indicates the row number counting from the
	 *        beginning of the result set; a negative number indicates the row
	 *        number counting from the end of the result set
	 * @return true if the cursor is on the result set; false otherwise
	 * @throws SQLException if a database access error occurs, or the result set
	 *         type is TYPE_FORWARD_ONLY
	 */
 	public boolean absolute(int row) throws SQLException {
		if (row != curRow + 1 && type == TYPE_FORWARD_ONLY) throw
			new SQLException("(Absolute) positioning not allowed on forward " +
				" only result sets!", "M1M05");

		if (header.isClosed())
			throw new SQLException("ResultSet is closed!", "M1M20");

		// first calculate what the JDBC row is
		if (row < 0) {
			// calculate the negatives...
			row = tupleCount + row + 1;
		}
		// now place the row not farther than just before or after the result
		if (row < 0) row = 0;	// before first
		else if (row > tupleCount + 1) row = tupleCount + 1;	// after last

		String tmpLine = header.getLine(row - 1);

		// store it
		curRow = row;

		if (tmpLine == null) return false;
		try {
			tlp.parse(tmpLine);
		} catch (MCLParseException e) {
			throw new SQLException(e.getMessage(), "M0M10");
		}

		// reset lastColumnRead
		lastColumnRead = -1;

		return true;
	}

	/**
	 * Moves the cursor to the end of this ResultSet object, just after the last
	 * row. This method has no effect if the result set contains no rows.
	 *
	 * @throws SQLException if a database access error occurs or the result set
	 *         type is TYPE_FORWARD_ONLY
	 */
	public void afterLast() throws SQLException {
		absolute(tupleCount + 1);
	}

	/**
	 * Moves the cursor to the front of this ResultSet object, just before the
	 * first row. This method has no effect if the result set contains no rows.
	 *
	 * @throws SQLException if a database access error occurs or the result set
	 *         type is TYPE_FORWARD_ONLY
	 */
	public void beforeFirst() throws SQLException {
		absolute(0);
	}

	/**
	 * Clears all warnings reported for this ResultSet object. After a call to
	 * this method, the method getWarnings returns null until a new warning is
	 * reported for this ResultSet object.
	 */
	public void clearWarnings() {
		warnings = null;
	}

	/**
	 * Releases this ResultSet object's database (and JDBC) resources
	 * immediately instead of waiting for this to happen when it is
	 * automatically closed.
	 */
	public void close() {
		if (!header.isClosed()) {
			header.close();
		}
		if (statement instanceof MonetStatement)
			((MonetStatement)statement).closeIfCompletion();
	}

	// Chapter 14.2.3 from Sun JDBC 3.0 specification
	/**
	 * Maps the given ResultSet column name to its ResultSet column index.
	 * Column names supplied to getter methods are case insensitive. If a select
	 * list contains the same column more than once, the first instance of the
	 * column will be returned.
	 *
	 * @param columnName the name of the column
	 * @return the column index of the given column name
	 * @throws SQLException if the ResultSet object does not contain columnName
	 */
	public int findColumn(String columnName) throws SQLException {
		if (columnName != null) {
			for (int i = 0; i < columns.length; i++) {
				if (columns[i].equals(columnName))
					return i + 1;
			}
			/* if an exact match did not succeed try a case insensitive match */
			for (int i = 0; i < columns.length; i++) {
				if (columns[i].equalsIgnoreCase(columnName))
					return i + 1;
			}
		}
		throw new SQLException("No such column name: " + columnName, "M1M05");
	}

	/**
	 * Moves the cursor to the first row in this ResultSet object.
	 *
	 * @return true if the cursor is on a valid row; false if there are no rows
	 *         in the result set
	 * @throws SQLException - if a database access error occurs or the result
	 *         set type is TYPE_FORWARD_ONLY
	 */
	public boolean first() throws SQLException {
		return absolute(1);
	}

	public Array getArray(int i) throws SQLException {
		throw new SQLFeatureNotSupportedException("Method getArray not implemented yet, sorry!", "0A000");
	}
	public Array getArray(String colName) throws SQLException {
		throw new SQLFeatureNotSupportedException("Method getArray not implemented yet, sorry!", "0A000");
	}

	/* Mapi doesn't allow something for streams at the moment, thus all not implemented for now */
	public InputStream getAsciiStream(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException("Method getAsciiStream not implemented yet, sorry!", "0A000");
	}
	public InputStream getAsciiStream(String columnName) throws SQLException {
		throw new SQLFeatureNotSupportedException("Method getAsciiStream not implemented yet, sorry!", "0A000");
	}
	public InputStream getUnicodeStream(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException("Method getUnicodeStream not implemented yet, sorry!", "0A000");
	}
	public InputStream getUnicodeStream(String columnName) throws SQLException {
		throw new SQLFeatureNotSupportedException("Method getUnicodeStream not implemented yet, sorry!", "0A000");
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a stream of uninterpreted bytes. The
	 * value can then be read in chunks from the stream. This method is
	 * particularly suitable for retrieving large LONGVARBINARY values.
	 * <br/><br/>
	 * Note: All the data in the returned stream must be read prior to
	 * getting the value of any other column. The next call to a getter
	 * method implicitly closes the stream. Also, a stream may return 0
	 * when the method InputStream.available  is called whether there is
	 * data available or not. 
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return a Java input stream that delivers the database column
	 * value as a stream of uninterpreted bytes; if the value is SQL
	 * NULL, the value returned is null
	 * @throws SQLException if the columnIndex is not valid; if a
	 * database access error occurs or this method is called on a closed
	 * result set
	 */
	public InputStream getBinaryStream(int columnIndex) throws SQLException {
		switch (getJavaType(types[columnIndex - 1])) {
			case Types.BLOB:
				Blob blob = getBlob(columnIndex);
				if (blob == null)
					return null;
				return blob.getBinaryStream();
			case Types.BINARY:
			case Types.VARBINARY:
			case Types.LONGVARBINARY:
				byte[] bte = getBytes(columnIndex);
				if (bte == null)
					return null;
				return new ByteArrayInputStream(bte);
		}
		throw new SQLException("Cannot operate on " +
				types[columnIndex - 1] + " type", "M1M05");
	}
	
	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a stream of uninterpreted bytes. The
	 * value can then be read in chunks from the stream. This method is
	 * particularly suitable for retrieving large LONGVARBINARY  values.
	 * <br/><br/>
	 * Note: All the data in the returned stream must be read prior to
	 * getting the value of any other column. The next call to a getter
	 * method implicitly closes the stream. Also, a stream may return 0
	 * when the method available  is called whether there is data
	 * available or not.
	 *
	 * @param columnLabel the label for the column specified with
	 * the SQL AS clause. If the SQL AS clause was not specified, then
	 * the label is the name of the column 
	 * @return a Java input stream that delivers the database column
	 * value as a stream of uninterpreted bytes; if the value is SQL
	 * NULL, the result is null
	 * @throws SQLException if the columnLabel is not valid; if a
	 * database access error occurs or this method is called on a closed
	 * result set
	 */
	public InputStream getBinaryStream(String columnName) throws SQLException {
		return getBinaryStream(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a java.io.Reader object.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return a java.io.Reader object that contains the column value;
	 *         if the value is SQL NULL, the value returned is null in
	 *         the Java programming language. 
	 * @throws SQLException if a database access error occurs
	 */
	public Reader getCharacterStream(int columnIndex) throws SQLException {
		String tmp = getString(columnIndex);
		if (tmp == null)
			return null;
		return new StringReader(tmp);
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a java.io.Reader object.
	 *
	 * @param columnName the name of the column
	 * @return a java.io.Reader object that contains the column value;
	 *         if the value is SQL NULL, the value returned is null in
	 *         the Java programming language. 
	 * @throws SQLException if a database access error occurs
	 */
	public Reader getCharacterStream(String columnName) throws SQLException {
		return getCharacterStream(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a java.io.Reader object. It is
	 * intended for use when accessing NCHAR,NVARCHAR and LONGNVARCHAR
	 * columns.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return a java.io.Reader object that contains the column value;
	 *         if the value is SQL NULL, the value returned is null in
	 *         the Java programming language. 
	 * @throws SQLException if a database access error occurs
	 * @throws SQLFeatureNotSupportedException the JDBC driver does
	 *         not support this method
	 */
	public Reader getNCharacterStream(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException("getNCharacterStream() not supported", "0A000");
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a java.io.Reader object. It is
	 * intended for use when accessing NCHAR,NVARCHAR and LONGNVARCHAR
	 * columns.
	 *
	 * @param columnName the name of the column
	 * @return a java.io.Reader object that contains the column value;
	 *         if the value is SQL NULL, the value returned is null in
	 *         the Java programming language. 
	 * @throws SQLException if a database access error occurs
	 * @throws SQLFeatureNotSupportedException the JDBC driver does
	 *         not support this method
	 */
	public Reader getNCharacterStream(String columnName) throws SQLException {
		return getNCharacterStream(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a Blob object in the Java programming
	 * language.
	 *
	 * @param i the first column is 1, the second is 2, ...
	 * @return a Blob object representing the SQL BLOB value in the
	 *         specified column
	 * @throws SQLException if a database access error occurs
	 */
	public Blob getBlob(int i) throws SQLException {
		String tmp = getString(i);
		if (tmp == null) {
			return null;
		} else {
			return MonetBlob.create(tmp);
		}
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a Blob object in the Java programming
	 * language.
	 *
	 * @param colName the name of the column from which to retrieve
	 *        the value
	 * @return a Blob object representing the SQL BLOB value in the
	 *         specified column
	 * @throws SQLException if a database access error occurs
	 */
	public Blob getBlob(String colName) throws SQLException {
		return getBlob(findColumn(colName));
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a Clob object in the
	 * Java programming language.
	 *
	 * @param i the first column is 1, the second is 2, ...
	 * @return a Clob object representing the SQL CLOB value in the
	 *         specified column
	 * @throws SQLException if a database access error occurs
	 */
	public Clob getClob(int i) throws SQLException {
		String tmp = getString(i);
		if (tmp == null) {
			return null;
		} else {
			return new MonetClob(tmp);
		}
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a Clob object in the
	 * Java programming language.
	 *
	 * @param colName the name of the column from which to retrieve
	 *        the value
	 * @return a Clob object representing the SQL CLOB value in the
	 *         specified column
	 * @throws SQLException if a database access error occurs
	 */
	public Clob getClob(String colName) throws SQLException {
		return getClob(findColumn(colName));
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a NClob object in the
	 * Java programming language.
	 *
	 * @param i the first column is 1, the second is 2, ...
	 * @return a NClob object representing the SQL NCLOB value in the
	 *         specified column
	 * @throws SQLException if a database access error occurs
	 * @throws SQLFeatureNotSupportedException the JDBC driver does
	 *         not support this method
	 */
	public NClob getNClob(int i) throws SQLException {
		throw new SQLFeatureNotSupportedException("getNClob() not supported", "0A000");
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a NClob object in the
	 * Java programming language.
	 *
	 * @param colName the name of the column from which to retrieve
	 *        the value
	 * @return a NClob object representing the SQL NCLOB value in the
	 *         specified column
	 * @throws SQLException if a database access error occurs
	 * @throws SQLFeatureNotSupportedException the JDBC driver does
	 *         not support this method
	 */
	public NClob getNClob(String colName) throws SQLException {
		return getNClob(findColumn(colName));
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.math.BigDecimal with full precision.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value (full precision); if the value is SQL NULL,
	 *         the value returned is null in the Java programming language.
	 * @throws SQLException if a database access error occurs
	 */
	public BigDecimal getBigDecimal(int columnIndex) throws SQLException {
		String decimal = getString(columnIndex);
		if (decimal == null) {
			return null;
		} else {
			try {
				return new BigDecimal(decimal);
			} catch (NumberFormatException e) {
				return BigDecimal.ZERO;
			}
		}
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.math.BigDecimal with full precision.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @param scale the number of digits to the right of the decimal point
	 * @return the column value (full precision); if the value is SQL NULL,
	 *         the value returned is null in the Java programming language.
	 * @throws SQLException if a database access error occurs
	 */
	public BigDecimal getBigDecimal(int columnIndex, int scale)
		throws SQLException
	{
		String decimal = getString(columnIndex);
		if (decimal == null) {
			return null;
		} else {
			BigDecimal bd;
			try {
				bd = new BigDecimal(decimal);
			} catch (NumberFormatException e) {
				bd = BigDecimal.ZERO;
			}
			return bd.setScale(scale);
		}
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.math.BigDecimal with full precision.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value (full precision); if the value is SQL NULL,
	 *         the value returned is null in the Java programming language.
	 * @throws SQLException if a database access error occurs
	 */
	public BigDecimal getBigDecimal(String columnName) throws SQLException {
		return getBigDecimal(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.math.BigDecimal with full precision.
	 *
	 * @param columnName the SQL name of the column
	 * @param scale the number of digits to the right of the decimal point
	 * @return the column value (full precision); if the value is SQL NULL,
	 *         the value returned is null in the Java programming language.
	 * @throws SQLException if a database access error occurs
	 */
	public BigDecimal getBigDecimal(String columnName, int scale)
		throws SQLException
	{
		return getBigDecimal(findColumn(columnName), scale);
	}

	// See Sun JDBC Specification 3.0 Table B-6
	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a boolean in the Java programming language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is false
	 * @throws SQLException if there is no such column
	 */
	public boolean getBoolean(int columnIndex) throws SQLException{
		int dataType = getJavaType(types[columnIndex - 1]);
		if (dataType == Types.TINYINT ||
			dataType == Types.SMALLINT ||
			dataType == Types.INTEGER ||
			dataType == Types.BIGINT)
		{
			if (getLong(columnIndex) == 0L) {
				return false;
			} else {
				return true;
			}
		} else if (dataType == Types.REAL ||
			dataType == Types.FLOAT ||
			dataType == Types.DOUBLE)
		{
			if (getDouble(columnIndex) == 0.0) {
				return false;
			} else {
				return true;
			}
		} else if (dataType == Types.DECIMAL ||
			dataType == Types.NUMERIC)
		{
			if (getBigDecimal(columnIndex).compareTo(new BigDecimal(0.0)) == 0) {
				return false;
			} else {
				return true;
			}
		} else if (dataType == Types.BIT ||
			dataType == Types.BOOLEAN ||
			dataType == Types.CHAR ||
			dataType == Types.VARCHAR ||
			dataType == Types.LONGVARCHAR)
		{
			return (Boolean.valueOf(getString(columnIndex))).booleanValue();
		} else {
			throw new SQLException("Conversion from " + types[columnIndex - 1] + " to boolean type not supported", "M1M05");
		}
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a boolean in the Java programming language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is false
	 * @throws SQLException if the ResultSet object does not contain columnName
	 */
	public boolean getBoolean(String columnName) throws SQLException {
		return getBoolean(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a byte in the Java programming language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if a database access error occurs
	 */
	public byte getByte(int columnIndex) throws SQLException {
		byte ret = 0;
		String val = getString(columnIndex);
		if (val != null) {
			ret = Byte.parseByte(val);
		}
		return ret;
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a byte in the Java programming language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if a database access error occurs
	 */
	public byte getByte(String columnName) throws SQLException {
		return getByte(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a byte array in the Java programming language. The
	 * bytes represent the raw values returned by the driver.
	 * <br /><br />
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if a database access error occurs
	 */
	public byte[] getBytes(int columnIndex) throws SQLException {
		// According to Table B-6, getBytes() only operates on BINARY
		// types
		switch (getJavaType(types[columnIndex - 1])) {
			case Types.BINARY:
			case Types.VARBINARY:
			case Types.LONGVARBINARY:
				// pass
				break;
			default:
				throw new SQLException("Cannot operate on " +
						types[columnIndex - 1] + " type", "M1M05");
		}
		String tmp = getString(columnIndex);
		if (tmp == null) {
			return null;
		} else {
			// unpack the HEX (BLOB) notation to real bytes
			int len = tmp.length() / 2;
			byte[] buf = new byte[len];
			for (int j = 0; j < len; j++)
				buf[j] = (byte)Integer.parseInt(tmp.substring(2 * j, (2 * j) + 2), 16);
			return buf;
		}
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a byte array in the Java programming language. The
	 * bytes represent the raw values returned by the driver.
	 * <br /><br />
	 * NOTE: Since the mapi protocol is ASCII-based, this method only returns
	 *       Java byte representations of Strings, which is nothing more than
	 *       an encoding into a sequence of bytes using the platform's default
	 *       charset.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if a database access error occurs
	 */
	public byte[] getBytes(String columnName) throws SQLException {
		return getBytes(findColumn(columnName));
	}

	/**
	 * Retrieves the concurrency mode of this ResultSet object. The concurrency
	 * used is determined by the Statement object that created the result set.
	 * <br /><br />
	 * NOTE: MonetDB only supports read-only result sets, and will always return
	 *       ResultSet.CONCUR_READ_ONLY
	 *
	 * @return the concurrency type, either ResultSet.CONCUR_READ_ONLY or
	 *         ResultSet.CONCUR_UPDATABLE
	 */
	public int getConcurrency() {
		return concurrency;
	}

	/**
	 * Retrieves the name of the SQL cursor used by this ResultSet object.
	 * In SQL, a result table is retrieved through a cursor that is named.
	 * For MonetDB this is the tableID returned in a resultset header. The
	 * current row of a result set can be updated or deleted using a positioned
	 * update/delete statement that references the cursor name. To insure that
	 * the cursor has the proper isolation level to support update, the
	 * cursor's SELECT statement should be of the form SELECT FOR UPDATE. If
	 * FOR UPDATE is omitted, the positioned updates may fail.
	 * <br /><br />
	 * The JDBC API supports this SQL feature by providing the name of the SQL
	 * cursor used by a ResultSet object. The current row of a ResultSet object
	 * is also the current row of this SQL cursor.
	 * <br /><br />
	 * Note: If positioned update is not supported, a SQLException is thrown.
	 *       MonetDB currently doesn't support updates, so the SQLException is
	 *       thrown for now.
	 *
	 * @return the SQL name for this ResultSet object's cursor
	 * @throws SQLException if a database access error occurs
	 */
	public String getCursorName() throws SQLException {
		throw new SQLException("Positioned updates not supported for this " +
							   "cursor (" + tableID + ")", "0AM21");
		//return "" + tableID;
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a double in the Java programming language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if there is no such column
	 */
	public double getDouble(int columnIndex) throws SQLException {
		double ret = 0;	// note: relaxing by compiler here
		String dbl = getString(columnIndex);
		if (dbl != null) {
			try {
				ret = Double.parseDouble(dbl);
			} catch (NumberFormatException e) {
				// ignore, return the default: 0
			}
			// do not catch SQLException for it is declared to be thrown
		}

		return ret;
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a double in the Java programming language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if the ResultSet object does not contain columnName
	 */
	public double getDouble(String columnName) throws SQLException {
		return getDouble(findColumn(columnName));
	}

	/**
	 * Retrieves the holdability of this ResultSet object.
	 *
	 * @return either ResultSet.HOLD_CURSORS_OVER_COMMIT or
	 *         ResultSet.CLOSE_CURSORS_AT_COMMIT
	 * @throws SQLException if a database access error occurs
	 */
	public int getHoldability() throws SQLException {
		return getStatement().getConnection().getHoldability();
	}

	/**
	 * Retrieves the fetch direction for this ResultSet object.
	 * <b>currently not implemented</b>
	 *
	 * @return the current fetch direction for this ResultSet object
	 */
	public int getFetchDirection() {
		return FETCH_FORWARD;
	}

	/**
	 * Retrieves the fetch size for this ResultSet object.
	 *
	 * @return the current fetch size for this ResultSet object
	 * @throws SQLException if a database access error occurs
	 */
	public int getFetchSize() throws SQLException {
		return header.getCacheSize();
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a float in the Java programming language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if there is no such column
	 */
	public float getFloat(int columnIndex) throws SQLException {
		float ret = 0;	// note: relaxing by compiler here
		String flt = getString(columnIndex);
		if (flt != null) {
			try {
				ret = Float.parseFloat(flt);
			} catch (NumberFormatException e) {
				// ignore, return the default: 0
			}
			// do not catch SQLException for it is declared to be thrown
		}

		return ret;
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a float in the Java programming language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if the ResultSet object does not contain columnName
	 */
	public float getFloat(String columnName) throws SQLException {
		return getFloat(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as an int in the Java programming language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if there is no such column
	 */
	public int getInt(int columnIndex) throws SQLException {
		int ret = 0;
		try {
			// note: Integer.parseInt DOES unlike Double and Float
			// accept a null value
			ret = Integer.parseInt(getString(columnIndex));
		} catch (NumberFormatException e) {
			// ignore, return the default: 0
		}
		// do not catch SQLException for it is declared to be thrown

		return ret;
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as an int in the Java programming language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if the ResultSet object does not contain columnName
	 */
	public int getInt(String columnName) throws SQLException {
		return getInt(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a long in the Java programming language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if there is no such column
	 */
	public long getLong(int columnIndex) throws SQLException {
		long ret = 0;
		try {
			// note: Long.parseLong DOES unlike Double and Float
			// accept a null value
			ret = Long.parseLong(getString(columnIndex));
		} catch (NumberFormatException e) {
			// ignore, return the default: 0
		}
		// do not catch SQLException for it is declared to be thrown

		return ret;
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a long in the Java programming language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if the ResultSet object does not contain columnName
	 */
	public long getLong(String columnName) throws SQLException {
		return getLong(findColumn(columnName));
	}

	/* helper for the anonymous class inside getMetaData */
	private abstract class rsmdw extends MonetWrapper implements ResultSetMetaData {}
	/**
	 * Retrieves the number, types and properties of this ResultSet object's
	 * columns.
	 *
	 * @return the description of this ResultSet object's columns
	 */
	public ResultSetMetaData getMetaData() {
		// return inner class which implements the ResultSetMetaData interface
		return new rsmdw() {
			// for the more expensive methods, we provide a simple cache
			// for the most expensive part; getting the ResultSet which
			// contains the data
			private DatabaseMetaData dbmd = null;
			private ResultSet[] colrs = new ResultSet[columns.length];

			/**
			 * Returns the number of columns in this ResultSet object.
			 *
			 * @returns the number of columns
			 */
			public int getColumnCount() {
				return columns.length;
			}

			/**
			 * Indicates whether the designated column is automatically
			 * numbered, thus read-only.
			 * 
			 * @param column the first column is 1, the second is 2, ...
			 * @return true if so; false otherwise
			 * @throws SQLException if a database access error occurs
			 */
			public boolean isAutoIncrement(int column) throws SQLException {
				// the only column I know of is a 'secret' column called rowid
				// with datatype oid
				// avoid nullpointer exception here
				if ("oid".equals(getColumnTypeName(column))) {
					return true;
				} else {
					return false;
				}
			}

			/**
			 * Indicates whether a column's case matters. This holds for all
			 * columns in MonetDB resultsets since the mapping is done case
			 * insensitive, therefore this method will always return false.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @returns false
			 */
			public boolean isCaseSensitive(int column) {
				return false;
			}

			/**
			 * Indicates whether the designated column can be used in a
			 * where clause.
			 * It is unknown to me what kind ot columns they regard to,
			 * as I think all columns are useable in a where clause.
			 * Returning true for all here, for the time being.
			 * Possible thought; maybe they want to know here if it's a
			 * real column existing in a table or not...
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @returns true
			 */
			public boolean isSearchable(int column) {
				return true;
			}

			/**
			 * Indicates whether the designated column is a cash value.
			 * From the MonetDB database perspective it is by definition
			 * unknown whether the value is a currency, because there are
			 * no currency datatypes such as MONEY.  With this knowledge
			 * we can always return false here.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @returns false
			 */
			public boolean isCurrency(int column) {
				return false;
			}
			
			/**
			 * Indicates whether values in the designated column are signed
			 * numbers.
			 * Within MonetDB all numeric types are signed.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return true if so; false otherwise
			 */
			public boolean isSigned(int column) throws SQLException {
				// we can hardcode this, based on the colum type
				switch (getColumnType(column)) {
					case Types.NUMERIC:
					case Types.DECIMAL:
					case Types.TINYINT:
					case Types.SMALLINT:
					case Types.INTEGER:
					case Types.BIGINT:
					case Types.REAL:
					case Types.FLOAT:
					case Types.DOUBLE:
						return true;
					case Types.BIT: // we don't use type BIT, it's here for completeness
					case Types.BOOLEAN:
					case Types.DATE:
					case Types.TIME:
					case Types.TIMESTAMP:
					default:
						return false;
				}
			}

			/**
			 * Indicates the designated column's normal maximum width in
			 * characters.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return the normal maximum number of characters allowed as the
			 *         width of the designated column
			 * @throws SQLException if there is no such column
			 */
			public int getColumnDisplaySize(int column) throws SQLException {
				int ret;
				try {
					ret = header.getColumnLengths()[column - 1];
				} catch (IndexOutOfBoundsException e) {
					throw new SQLException("No such column " + column, "M1M05");
				}
				
				return ret;
			}

			/**
			 * Get the designated column's table's schema.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return schema name or "" if not applicable
			 * @throws SQLException if a database access error occurs
			 */
			public String getSchemaName(int column) throws SQLException {
				String schema = "";
				
				// figure the name out
				try {
					schema = header.getTableNames()[column - 1];
				} catch (IndexOutOfBoundsException e) {
					throw new SQLException("No such column " + column, "M1M05");
				}

				if (schema == null) throw
					new AssertionError("table_name header is empty!");
				int dot = schema.indexOf(".");
				if (dot == -1) throw
					new AssertionError("table_name is not fully qualified! (" + schema + ")");

				return schema.substring(0, dot);
			}

			/**
			 * Gets the designated column's table name.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return table name or "" if not applicable
			 */
			public String getTableName(int column) throws SQLException {
				String table = "";
				
				// figure the name out
				try {
					table = header.getTableNames()[column - 1];
				} catch (IndexOutOfBoundsException e) {
					throw new SQLException("No such column " + column, "M1M05");
				}

				if (table == null) throw
					new AssertionError("table_name header is empty!");
				int dot = table.indexOf(".");
				if (dot == -1) throw
					new AssertionError("table_name is not fully qualified! (" + table + ")");

				return table.substring(dot + 1);
			}

			/**
			 * Get the designated column's number of decimal digits.
			 * This method is currently very expensive as it needs to
			 * retrieve the information from the database using an SQL
			 * query.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return precision
			 * @throws SQLException if a database access error occurs
			 */
			public int getPrecision(int column) throws SQLException {
				int precision = 0;
				ResultSet col = getColumnResultSet(column);

				// the result has either zero or one results, as the
				// schema, table and column should be unique...
				if (col.next()) precision = col.getInt("COLUMN_SIZE");

				return precision;
			}

			/**
			 * Gets the designated column's number of digits to right of
			 * the decimal point.  This method is currently very
			 * expensive as it needs to retrieve the information from
			 * the database using an SQL query.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return scale
			 * @throws SQLException if a database access error occurs
			 */
			public int getScale(int column) throws SQLException {
				int scale = 0;
				ResultSet col = getColumnResultSet(column);

				// the result has either zero or one results, as the
				// schema, table and column should be unique...
				if (col.next()) scale = col.getInt("DECIMAL_DIGITS");

				return scale;
			}

			/**
			 * Indicates the nullability of values in the designated
			 * column.  This method is currently very expensive as it
			 * needs to retrieve the information from the database using
			 * an SQL query.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return scale
			 * @throws SQLException if a database access error occurs
			 */
			public int isNullable(int column) throws SQLException {
				int ret = columnNullableUnknown;
				ResultSet col = getColumnResultSet(column);

				// the result has either zero or one results, as the
				// schema, table and column should be unique...
				if (col.next()) ret = col.getInt("NULLABLE");

				return ret;
			}

			/**
			 * Gets the designated column's table's catalog name.
			 * Because MonetDB handles only one catalog (dbfarm) at a
			 * time, the current one is the one we deal with here.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return the name of the catalog for the table in which the given
			 *         column appears or "" if not applicable
			 */
			public String getCatalogName(int column) throws SQLException {
				if (getTableName(column) != "") {
					return getStatement().getConnection().getCatalog();
				} else {
					return "";
				}
			}

			/**
			 * Indicates whether the designated column is definitely not
			 * writable.  MonetDB does not support cursor updates, so
			 * nothing is writable.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return true if so; false otherwise
			 */
			public boolean isReadOnly(int column) {
				return true;
			}

			/**
			 * Indicates whether it is possible for a write on the
			 * designated column to succeed.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return true if so; false otherwise
			 */
			public boolean isWritable(int column) {
				return false;
			}

			/**
			 * Indicates whether a write on the designated column will
			 * definitely succeed.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return true if so; false otherwise
			 */
			public boolean isDefinitelyWritable(int column) {
				return false;
			}

			/**
			 * Returns the fully-qualified name of the Java class whose
			 * instances are manufactured if the method
			 * ResultSet.getObject is called to retrieve a value from
			 * the column.  ResultSet.getObject may return a subclass of
			 * the class returned by this method.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return the fully-qualified name of the class in the Java
			 *         programming language that would be used by the method
			 *         ResultSet.getObject to retrieve the value in the
			 *         specified column. This is the class name used for custom
			 *         mapping.
			 * @throws SQLException if there is no such column
			 */
			public String getColumnClassName(int column) throws SQLException {
				try {
					Class type;
					Map map = getStatement().getConnection().getTypeMap();
					if (map.containsKey(types[column - 1])) {
						type = (Class)map.get(types[column - 1]);
					} else {
						type = getClassForType(getJavaType(types[column - 1]));
					}
					if (type != null)
						return type.getName();
					throw new SQLException("column type mapping null: " +
							types[column - 1], "M0M03");
				} catch (IndexOutOfBoundsException e) {
					throw new SQLException("No such column " + column, "M1M05");
				}
			}

			/**
			 * Gets the designated column's suggested title for use in
			 * printouts and displays. This is currently equal to
			 * getColumnName().
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return the suggested column title
			 * @throws SQLException if there is no such column
			 */
			public String getColumnLabel(int column) throws SQLException {
				return getColumnName(column);
			}

			/**
			 * Gets the designated column's name
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return the column name
			 * @throws SQLException if there is no such column
			 */
			public String getColumnName(int column) throws SQLException {
				try {
					return columns[column - 1];
				} catch (IndexOutOfBoundsException e) {
					throw new SQLException("No such column " + column, "M1M05");
				}
			}

			/**
			 * Retrieves the designated column's SQL type.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return SQL type from java.sql.Types
			 * @throws SQLException if there is no such column
			 */
			public int getColumnType(int column) throws SQLException {
				String type = getColumnTypeName(column);

				return getJavaType(type);
			}

			/**
			 * Retrieves the designated column's database-specific type name.
			 *
			 * @param column the first column is 1, the second is 2, ...
			 * @return type name used by the database. If the column type is a
			 *         user-defined type, then a fully-qualified type name is
			 *         returned.
			 * @throws SQLException if there is no such column
			 */
			public String getColumnTypeName(int column) throws SQLException {
				try {
					return types[column - 1];
				} catch (IndexOutOfBoundsException e) {
					throw new SQLException("No such column " + column, "M1M05");
				}
			}

			/**
			 * Returns the Metadata ResultSet for the given column
			 * number of this ResultSet.  If the column was previously
			 * requested, a cached ResultSet is returned, otherwise it
			 * is fetched using the DatabaseMetaData class.
			 *
			 * @param column the column index number starting from 1
			 * @return Metadata ResultSet
			 * @throws SQLException if a database error occurs
			 */
			private ResultSet getColumnResultSet(int column)
				throws SQLException
			{
				if (column > columns.length || column <= 0) throw
					new SQLException("No such column " + column, "M1M05");

				if (colrs[column - 1] == null) {
					if (dbmd == null)
						dbmd = getStatement().getConnection().getMetaData();
					ResultSet col = 
						dbmd.getColumns(
								null, /* this doesn't matter here... */
								getSchemaName(column),
								getTableName(column),
								getColumnName(column)
							);
					colrs[column - 1] = col;
				}

				colrs[column - 1].beforeFirst();
				return colrs[column - 1];
			}
		};
	}

	/**
	 * Gets the value of the designated column in the current row of this
	 * ResultSet object as an Object in the Java programming language.
	 * <br /><br />
	 * This method will return the value of the given column as a Java object.
	 * The type of the Java object will be the default Java object type
	 * corresponding to the column's SQL type, following the mapping for
	 * built-in types specified in the JDBC specification. If the value is
	 * an SQL NULL, the driver returns a Java null.
	 * <br /><br />
	 * This method may also be used to read database-specific abstract data
	 * types. In the JDBC 2.0 API, the behavior of method getObject is extended
	 * to materialize data of SQL user-defined types. When a column contains a
	 * structured or distinct value, the behavior of this method is as if it
	 * were a call to: getObject(columnIndex,
	 * this.getStatement().getConnection().getTypeMap()).
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return a java.lang.Object holding the column value
	 * @throws SQLException if a database access error occurs
	 */
	public Object getObject(int columnIndex) throws SQLException {
		return getObject(columnIndex, this.getStatement().getConnection().getTypeMap());
	}

	private boolean classImplementsSQLData(Class cl) {
		Class[] cls = cl.getInterfaces();
		for (int i = 0; i < cls.length; i++) {
			if (cls[i] == SQLData.class)
				return true;
		}
		return false;
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as an Object in the Java programming language. If the
	 * value is an SQL NULL, the driver returns a Java null. This method uses
	 * the given Map object for the custom mapping of the SQL structured or
	 * distinct type that is being retrieved.
	 *
	 * @param i the first column is 1, the second is 2, ...
	 * @param map a java.util.Map object that contains the mapping from SQL
	 *        type names to classes in the Java programming language
	 * @return an Object in the Java programming language representing the SQL
	 *         value
	 * @throws SQLException if a database access error occurs
	 */
	public Object getObject(int i, Map<String,Class<?>> map)
		throws SQLException
	{
		Class<?> type;

		if (tlp.values[i - 1] == null) {
			lastColumnRead = i - 1;
			return null;
		}

		if (map.containsKey(types[i - 1])) {
			type = map.get(types[i - 1]);
		} else {
			type = getClassForType(getJavaType(types[i - 1]));
		}

		if (type == String.class) {
			return getString(i);
		} else if (type == BigDecimal.class) {
			return getBigDecimal(i);
		} else if (type == Boolean.class) {
			return Boolean.valueOf(getBoolean(i));
		} else if (type == Integer.class) {
			return Integer.valueOf(getInt(i));
		} else if (type == Long.class) {
			return Long.valueOf(getLong(i));
		} else if (type == Float.class) {
			return Float.valueOf(getFloat(i));
		} else if (type == Double.class) {
			return Double.valueOf(getDouble(i));
		} else if (type == byte[].class) {
			return getBytes(i);
		} else if (type == java.sql.Date.class) {
			return getDate(i);
		} else if (type == Time.class) {
			return getTime(i);
		} else if (type == Timestamp.class) {
			return getTimestamp(i);
		} else if (type == Clob.class) {
			return getClob(i);
		} else if (type == Blob.class) {
			return getBlob(i);
		} else if (classImplementsSQLData(type)) {
			SQLData x;
			try {
				Constructor<? extends SQLData> ctor =
					((Class)type).getConstructor();
				x = ctor.newInstance();
			} catch (NoSuchMethodException nsme) {
				throw new SQLException(nsme.getMessage(), "M0M27");
			} catch (InstantiationException ie) {
				throw new SQLException(ie.getMessage(), "M0M27");
			} catch (IllegalAccessException iae) {
				throw new SQLException(iae.getMessage(), "M0M27");
			} catch (IllegalArgumentException ige) {
				throw new SQLException(ige.getMessage(), "M0M27");
			} catch (InvocationTargetException ite) {
				throw new SQLException(ite.getMessage(), "M0M27");
			}
			final int colnum = i;
			final boolean valwasnull = wasNull();
			SQLInput input = new SQLInput() {
				public String readString() throws SQLException {
					return getString(colnum);
				}

				public boolean readBoolean() throws SQLException {
					return getBoolean(colnum);
				}

				public byte readByte() throws SQLException {
					return getByte(colnum);
				}

				public short readShort() throws SQLException {
					return getShort(colnum);
				}

				public int readInt() throws SQLException {
					return getInt(colnum);
				}

				public long readLong() throws SQLException {
					return getLong(colnum);
				}

				public float readFloat() throws SQLException {
					return getFloat(colnum);
				}

				public double readDouble() throws SQLException {
					return getDouble(colnum);
				}

				public BigDecimal readBigDecimal() throws SQLException {
					return getBigDecimal(colnum);
				}

				public byte[] readBytes() throws SQLException {
					return getBytes(colnum);
				}

				public java.sql.Date readDate() throws SQLException {
					return getDate(colnum);
				}

				public java.sql.Time readTime() throws SQLException {
					return getTime(colnum);
				}

				public Timestamp readTimestamp() throws SQLException {
					return getTimestamp(colnum);
				}

				public Reader readCharacterStream() throws SQLException {
					return getCharacterStream(colnum);
				}

				public InputStream readAsciiStream() throws SQLException {
					return getAsciiStream(colnum);
				}

				public InputStream readBinaryStream() throws SQLException {
					return getBinaryStream(colnum);
				}

				public Object readObject() throws SQLException {
					return getObject(colnum);
				}

				public Ref readRef() throws SQLException {
					return getRef(colnum);
				}

				public Blob readBlob() throws SQLException {
					return getBlob(colnum);
				}

				public Clob readClob() throws SQLException {
					return getClob(colnum);
				}

				public Array readArray() throws SQLException {
					return getArray(colnum);
				}

				public boolean wasNull() throws SQLException {
					return valwasnull;
				}

				public URL readURL() throws SQLException {
					return getURL(colnum);
				}

				public NClob readNClob() throws SQLException {
					return getNClob(colnum);
				}

				public String readNString() throws SQLException {
					return getNString(colnum);
				}

				public SQLXML readSQLXML() throws SQLException {
					return getSQLXML(colnum);
				}

				public RowId readRowId() throws SQLException {
					return getRowId(colnum);
				}
			};
			x.readSQL(input, types[i - 1]);
			return x;
		} else {
			return getString(i);
		}
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object and will convert from the SQL type of
	 * the column to the requested Java data type, if the conversion is
	 * supported.  If the conversion is not supported or null is
	 * specified for the type, a SQLException is thrown.
	 *
	 * @param i the first column is 1, the second is 2, ...
	 * @param type Class representing the Java data type to convert the
	 *        designated column to
	 * @return an instance of type holding the column value
	 * @throws SQLException if conversion is not supported, type is
	 *         null or another error occurs. The getCause() method of
	 *         the exception may provide a more detailed exception, for
	 *         example, if a conversion error occurs
	 */
	public <T> T getObject(int i, Class<T> type) throws SQLException {
		if (type == null)
			throw new SQLException("type is null", "M1M05");

		throw new SQLFeatureNotSupportedException("cannot return a Java generic type based on static types from getXXX methods", "0AM34");
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object and will convert from the SQL type of
	 * the column to the requested Java data type, if the conversion is
	 * supported.  If the conversion is not supported or null is
	 * specified for the type, a SQLException is thrown.
	 *
	 * @param columnLabel the label for the column specified with the
	 *        SQL AS clause. If the SQL AS clause was not specified,
	 *        then the label is the name of the column
	 * @param type Class representing the Java data type to convert the
	 *        designated column to
	 * @return an instance of type holding the column value
	 * @throws SQLException if conversion is not supported, type is
	 *         null or another error occurs. The getCause() method of
	 *         the exception may provide a more detailed exception, for
	 *         example, if a conversion error occurs
	 */
	public <T> T getObject(String columnLabel, Class<T> type)
		throws SQLException
	{
		return getObject(findColumn(columnLabel), type);
	}

	/**
	 * Helper method to support the getObject and
	 * ResultsetMetaData.getColumnClassName JDBC methods.
	 *
	 * @param type a value from java.sql.Types
	 * @return a Class object from which an instance would be returned
	 */
	static Class<?> getClassForType(int type) {
		/**
		 * This switch returns the types as objects according to table B-3 from
		 * Oracle's JDBC specification 4.1
		 */
		// keep this switch aligned with getObject(int, Map) !
		switch(type) {
			case Types.CHAR:
			case Types.VARCHAR:
			case Types.LONGVARCHAR:
				return String.class;
			case Types.NUMERIC:
			case Types.DECIMAL:
				return BigDecimal.class;
			case Types.BIT: // we don't use type BIT, it's here for completeness
			case Types.BOOLEAN:
				return Boolean.class;
			case Types.TINYINT:
			case Types.SMALLINT:
				return Short.class;
			case Types.INTEGER:
				return Integer.class;
			case Types.BIGINT:
				return Long.class;
			case Types.REAL:
				return Float.class;
			case Types.FLOAT:
			case Types.DOUBLE:
				return Double.class;
			case Types.BINARY:      // MonetDB currently does not support these
			case Types.VARBINARY:   // see treat_blob_as_binary property
			case Types.LONGVARBINARY:
				return byte[].class;
			case Types.DATE:
				return java.sql.Date.class;
			case Types.TIME:
				return Time.class;
			case Types.TIMESTAMP:
				return Timestamp.class;
			case Types.CLOB:
				return Clob.class;
			case Types.BLOB:
				return Blob.class;

			// all the rest are currently not implemented and used
			default:
				return String.class;
		}
	}

	/**
	 * Gets the value of the designated column in the current row of this
	 * ResultSet object as an Object in the Java programming language.
	 * <br /><br />
	 * This method will return the value of the given column as a Java object.
	 * The type of the Java object will be the default Java object type
	 * corresponding to the column's SQL type, following the mapping for
	 * built-in types specified in the JDBC specification. If the value is an
	 * SQL NULL, the driver returns a Java null.
	 * <br /><br />
	 * This method may also be used to read database-specific abstract data
	 * types.
	 * <br /><br />
	 * In the JDBC 2.0 API, the behavior of the method getObject is extended to
	 * materialize data of SQL user-defined types. When a column contains a
	 * structured or distinct value, the behavior of this method is as if it
	 * were a call to: getObject(columnName,
	 * this.getStatement().getConnection().getTypeMap()).
	 *
	 * @param columnName the SQL name of the column
	 * @return a java.lang.Object holding the column value
	 * @throws SQLException if a database access error occurs
	 */
	public Object getObject(String columnName) throws SQLException {
		return getObject(columnName, this.getStatement().getConnection().getTypeMap());
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as an Object  in the Java programming language. If the
	 * value is an SQL NULL, the driver returns a Java null. This method uses
	 * the specified Map object for custom mapping if appropriate.
	 *
	 * @param colName the name of the column from which to retrieve the value
	 * @param map a java.util.Map object that contains the mapping from SQL
	 *        type names to classes in the Java programming language
	 * @return an Object representing the SQL value in the specified column
	 * @throws SQLException if a database access error occurs
	 */
	public Object getObject(String colName, Map<String,Class<?>> map) throws SQLException {
		return getObject(findColumn(colName), map);
	}

	public Ref getRef(int i) throws SQLException { throw new SQLException("Method getRef not implemented yet, sorry!", "0A000"); }
	public Ref getRef(String colName) throws SQLException { throw new SQLException("Method getRef not implemented yet, sorry!", "0A000"); }

	/**
	 * Retrieves the current row number. The first row is number 1, the second
	 * number 2, and so on.
	 *
	 * @return the current row number; 0 if there is no current row
	 */
	public int getRow() {
		return curRow;
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a java.sql.RowId object in the Java
	 * programming language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if there is no such column
	 * @throws SQLFeatureNotSupportedException the JDBC driver does
	 *         not support this method
	 */
	public RowId getRowId(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException("getRowId() not supported", "0A000");
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a java.sql.RowId object in the Java
	 * programming language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if the ResultSet object does not contain columnName
	 * @throws SQLFeatureNotSupportedException the JDBC driver does
	 *         not support this method
	 */
	public RowId getRowId(String columnName) throws SQLException {
		return getRowId(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a short in the Java programming language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if there is no such column
	 */
	public short getShort(int columnIndex) throws SQLException {
		short ret = 0;	// note: relaxing by compiler here
		try {
			ret = Short.parseShort(getString(columnIndex));
		} catch (NumberFormatException e) {
			// ignore, return the default: 0
		}
		// do not catch SQLException for it is declared to be thrown

		return ret;
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a short in the Java programming language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is 0
	 * @throws SQLException if the ResultSet object does not contain columnName
	 */
	public short getShort(String columnName) throws SQLException {
		return getShort(findColumn(columnName));
	}

	/**
	 * Retrieves the Statement object that produced this ResultSet object. If
	 * the result set was generated some other way, such as by a
	 * DatabaseMetaData method, this method returns null.
	 *
	 * @return the Statment object that produced this ResultSet object or null
	 *         if the result set was produced some other way
	 */
	public Statement getStatement() {
		return statement;
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a String in the Java programming language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if there is no such column
	 */
	public String getString(int columnIndex) throws SQLException {
		// note: all current getters use the string getter in the end
		// in the future this might change, and the lastColumnRead must
		// be updated for the wasNull command to work properly!!!
		try {
			String ret = tlp.values[columnIndex - 1];
			lastColumnRead = columnIndex - 1;
			return ret;
		} catch (IndexOutOfBoundsException e) {
			throw new SQLException("No such column " + columnIndex, "M1M05");
		}
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a String in the Java programming language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if the ResultSet object does not contain columnName
	 */
	public String getString(String columnName) throws SQLException {
		return getString(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a String in the Java programming
	 * language. It is intended for use when accessing NCHAR,NVARCHAR
	 * and LONGNVARCHAR columns.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if there is no such column
	 * @throws SQLFeatureNotSupportedException the JDBC driver does
	 *         not support this method
	 */
	public String getNString(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException("getNString() not supported", "0A000");
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a String in the Java programming
	 * language. It is intended for use when accessing NCHAR,NVARCHAR
	 * and LONGNVARCHAR columns.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if the ResultSet object does not contain columnName
	 * @throws SQLFeatureNotSupportedException the JDBC driver does
	 *         not support this method
	 */
	public String getNString(String columnName) throws SQLException {
		return getNString(findColumn(columnName));
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet as a java.sql.SQLXML object in the Java
	 * programming language.
	 *
	 * @param i the first column is 1, the second is 2, ...
	 * @return a SQLXML object that maps an SQL XML value
	 * @throws SQLException if a database access error occurs
	 * @throws SQLFeatureNotSupportedException the JDBC driver does
	 *         not support this method
	 */
	public SQLXML getSQLXML(int i) throws SQLException {
		throw new SQLFeatureNotSupportedException("getSQLXML() not supported", "0A000");
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet as a java.sql.SQLXML object in the Java
	 * programming language. 
	 *
	 * @param colName the label for the column specified with the SQL AS
	 *        clause. If the SQL AS clause was not specified, then the
	 *        label is the name of the column
	 * @return a SQLXML object that maps an SQL XML value
	 * @throws SQLException if a database access error occurs
	 * @throws SQLFeatureNotSupportedException the JDBC driver does
	 *         not support this method
	 */
	public SQLXML getSQLXML(String colName) throws SQLException {
		return getSQLXML(findColumn(colName));
	}

	// This behaviour is according table B-6 of Sun JDBC Specification 3.0
	private SimpleDateFormat ts =
		new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	private SimpleDateFormat t =
		new SimpleDateFormat("HH:mm:ss");
	private SimpleDateFormat d =
		new SimpleDateFormat("yyyy-MM-dd");
	/**
	 * Helper method which parses the date/time value for columns of type
	 * TIME, DATE and TIMESTAMP.  For the types CHAR, VARCHAR and
	 * LONGVARCHAR an attempt is made to parse the date according to the
	 * given type.  The given Calender object is filled with the parsed
	 * data.  Optional fractional seconds (nanos) are returned by this
	 * method.  If the underlying type of the column is none of the
	 * mentioned six, January 1st 1970 0:00:00 GMT is returned.<br />
	 * The dates are parsed with the given Calendar.
	 *
	 * @param cal the Calendar to use/fill when parsing the date/time
	 * @param col the column to parse
	 * @param type the corresponding java.sql.Types type of the calling
	 *        function
	 * @return the fractional seconds (nanos) or -1 if the value is NULL
	 * @throws SQLException if a database error occurs
	 */
	private int getJavaDate(Calendar cal, int col, int type)
		throws SQLException
	{
		if (cal == null) throw
			new IllegalArgumentException("No Calendar object given!");
		if (col <= 0) throw
			new IllegalArgumentException("No valid column number given!");

		String monetDate = getString(col);
		if (monetDate == null)
			return -1;

		int nanos = 0;
		TimeZone ptz = cal.getTimeZone();

		// If we got a string type, set the datatype to the given
		// type so we attempt to parse it as the caller thinks it is.
		int dataType = getJavaType(types[col - 1]);
		if (dataType == Types.CHAR ||
			dataType == Types.VARCHAR ||
			dataType == Types.LONGVARCHAR)
		{
			dataType = type;
		}

		// we know whether we have a time with or without
		// time zone if the monet type ends with "tz"
		boolean hasTimeZone = types[col - 1].endsWith("tz");

		// it is important to parse the time in the given timezone in
		// order to get a correct (UTC) time value, hence we need to
		// parse it first
		if (hasTimeZone) {
			// MonetDB/SQL99:  Sign TwoDigitHours : Minutes
			ptz = TimeZone.getTimeZone("GMT" +
					monetDate.substring(
						monetDate.length() - 6,
						monetDate.length()));
		}
		ts.setTimeZone(ptz);
		t.setTimeZone(ptz);
		d.setTimeZone(ptz);

		java.util.Date pdate = null;
		ParsePosition ppos = new ParsePosition(0);
		switch(dataType) {
			default:
				addWarning("unsupported data type", "01M03");
				cal.clear();
				nanos = 0;
				return nanos;
			case Types.DATE:
				pdate = d.parse(monetDate, ppos);
				break;
			case Types.TIME:
				pdate = t.parse(monetDate, ppos);
				break;
			case Types.TIMESTAMP:
				pdate = ts.parse(monetDate, ppos);
				break;
		}
		if (pdate == null) {
			// parsing failed
			int epos = ppos.getErrorIndex();
			if (epos == -1) {
				addWarning("parsing '" + monetDate + "' failed", "01M10");
			} else if (epos < monetDate.length()) {
				addWarning("parsing failed," +
						 " found: '" + monetDate.charAt(epos) + "'" +
						 " in: \"" + monetDate + "\"" +
						 " at pos: " + ppos.getErrorIndex(), "01M10");
			} else {
				addWarning("parsing failed, expected more data after '" +
						monetDate + "'", "01M10");
			}
			// default value
			cal.clear();
			nanos = 0;
			return nanos;
		}
		cal.setTime(pdate);

		if (dataType == Types.TIME || dataType == Types.TIMESTAMP) {
			// parse additional nanos (if any)
			int pos = ppos.getIndex();
			char[] monDate = monetDate.toCharArray();
			if (pos < monDate.length && monDate[pos] == '.') {
				pos++;
				int ctr;
				try {
					nanos = getIntrinsicValue(monDate[pos], pos++);
					for (ctr = 1;
							pos < monDate.length && 
							monDate[pos] >= '0' &&
							monDate[pos] <= '9';
							ctr++)
					{
						if (ctr < 9) {
							nanos *= 10;
							nanos += (getIntrinsicValue(monDate[pos], pos));
						}
						if (ctr == 2)	// we have three at this point
							cal.set(Calendar.MILLISECOND, nanos);
						pos++;
					}
					while (ctr++ < 9)
						nanos *= 10;
				} catch(MCLParseException e) {
					addWarning(e.getMessage() +
							" found: '" + monDate[e.getErrorOffset()] + "'" +
							" in: \"" + monetDate + "\"" +
							" at pos: " + e.getErrorOffset(), "01M10");
					// default value
					cal.clear();
					nanos = 0;
				}
			}
		}
		return nanos;
	}

	/**
	 * Small helper method that returns the intrinsic value of a char if
	 * it represents a digit.  If a non-digit character is encountered
	 * an MCLParseException is thrown.
	 *
	 * @param c the char
	 * @param pos the position
	 * @return the intrinsic value of the char
	 * @throws MCLParseException if c is not a digit
	 */
	private final static int getIntrinsicValue(char c, int pos)
		throws MCLParseException
	{
		// note: don't use Character.isDigit() here, because
		// we only want ISO-LATIN-1 digits
		if (c >= '0' && c <= '9') {
			return (int)c - (int)'0';
		} else {
			throw new MCLParseException("Expected a digit", pos);
		}
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.sql.Date object in the Java programming
	 * language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if a database access error occurs
	 * @see #getDate(int col, Calendar cal)
	 */
	public java.sql.Date getDate(int columnIndex) throws SQLException {
		return getDate(columnIndex,	Calendar.getInstance());
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.sql.Date object in the Java programming
	 * language. This method uses the given calendar to construct an appropriate
	 * millisecond value for the date if the underlying database does not store
	 * timezone information.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @param cal the java.util.Calendar object to use in constructing the date
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if a database access error occurs
	 */
	public java.sql.Date getDate(int columnIndex, Calendar cal)
		throws SQLException
	{
		int ret = getJavaDate(cal, columnIndex, Types.DATE);
		return ret == -1 ? null : new java.sql.Date(cal.getTimeInMillis());
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.sql.Date object in the Java programming
	 * language.
	 *
	 * @param columnName the SQL name of the column from which to retrieve the
	 *        value
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if a database access error occurs
	 */
	public java.sql.Date getDate(String columnName) throws SQLException {
		return getDate(columnName, Calendar.getInstance());
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.sql.Date object in the Java programming
	 * language. This method uses the given calendar to construct an appropriate
	 * millisecond value for the date if the underlying database does not store
	 * timezone information.
	 *
	 * @param columnName the SQL name of the column from which to retrieve the
	 *        value
	 * @param cal the java.util.Calendar object to use in constructing the date
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if a database access error occurs
	 */
	public java.sql.Date getDate(String columnName, Calendar cal)
		throws SQLException
	{
		return getDate(findColumn(columnName), cal);
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.sql.Time object in the Java programming
	 * language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if a database access error occurs
	 */
	public Time getTime(int columnIndex) throws SQLException {
		return getTime(columnIndex, Calendar.getInstance());
	}

	/**
	 * Retrieves the value of the designated column in the current row of
	 * this ResultSet object as a java.sql.Time object in the Java programming
	 * language. This method uses the given calendar to construct an appropriate
	 * millisecond value for the time if the underlying database does not store
	 * timezone information.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @param cal the java.util.Calendar object to use in constructing the
	 *        timestamp
	 * @return the column value as a java.sql.Timestamp object; if the value is
	 *         SQL NULL, the value returned is null in the Java programming
	 *         language
	 * @throws SQLException if a database access error occurs
	 */
	public Time getTime(int columnIndex, Calendar cal)
		throws SQLException
	{
		int ret = getJavaDate(cal, columnIndex, Types.TIME);
		return ret == -1 ? null : new Time(cal.getTimeInMillis());
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.sql.Time object in the Java programming
	 * language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if a database access error occurs
	 */
	public Time getTime(String columnName) throws SQLException {
		return getTime(columnName, Calendar.getInstance());
	}

	/**
	 * Retrieves the value of the designated column in the current row of
	 * this ResultSet object as a java.sql.Time object in the Java programming
	 * language. This method uses the given calendar to construct an appropriate
	 * millisecond value for the time if the underlying database does not store
	 * timezone information.
	 *
	 * @param columnName the SQL name of the column
	 * @param cal the java.util.Calendar object to use in constructing the
	 *        timestamp
	 * @return the column value as a java.sql.Timestamp object; if the value is
	 *         SQL NULL, the value returned is null in the Java programming
	 *         language
	 * @throws SQLException if a database access error occurs
	 */
	public Time getTime(String columnName, Calendar cal)
		throws SQLException
	{
		return getTime(findColumn(columnName), cal);
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.sql.Timestamp object in the Java programming
	 * language.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if a database access error occurs
	 */
	public Timestamp getTimestamp(int columnIndex) throws SQLException {
		return getTimestamp(columnIndex, Calendar.getInstance());
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.sql.Timestamp object in the Java programming
	 * language. This method uses the given calendar to construct an appropriate
	 * millisecond value for the timestamp if the underlying database does not
	 * store timezone information.
	 *
	 * @param columnIndex the first column is 1, the second is 2, ...
	 * @param cal the java.util.Calendar object to use in constructing the
	 *        timestamp
	 * @return the column value as a java.sql.Timestamp object; if the value is
	 *         SQL NULL, the value returned is null in the Java programming
	 *         language
	 * @throws SQLException if a database access error occurs
	 */
	public Timestamp getTimestamp(int columnIndex, Calendar cal)
		throws SQLException
	{
		int nanos = getJavaDate(cal, columnIndex, Types.TIMESTAMP);
		if (nanos == -1) return null;
		Timestamp ts = new Timestamp(cal.getTimeInMillis());
		ts.setNanos(nanos);

		return ts;
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.sql.Timestamp object in the Java programming
	 * language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value; if the value is SQL NULL, the value returned
	 *         is null
	 * @throws SQLException if a database access error occurs
	 */
	public Timestamp getTimestamp(String columnName) throws SQLException {
		return getTimestamp(columnName, Calendar.getInstance());
	}

	/**
	 * Retrieves the value of the designated column in the current row of this
	 * ResultSet object as a java.sql.Timestamp object in the Java programming
	 * language. This method uses the given calendar to construct an appropriate
	 * millisecond value for the timestamp if the underlying database does not
	 * store timezone information.
	 *
	 * @param columnName the SQL name of the column
	 * @param cal the java.util.Calendar object to use in constructing the
	 *        timestamp
	 * @return the column value as a java.sql.Timestamp object; if the value is
	 *         SQL NULL, the value returned is null in the Java programming
	 *         language
	 * @throws SQLException if a database access error occurs
	 */
	public Timestamp getTimestamp(String columnName, Calendar cal)
		throws SQLException
	{
		return getTimestamp(findColumn(columnName), cal);
	}

	/**
	 * Retrieves the type of this ResultSet object. The type is determined by
	 * the Statement object that created the result set.
	 *
	 * @return ResultSet.TYPE_FORWARD_ONLY, ResultSet.TYPE_SCROLL_INSENSITIVE,
	 *         or ResultSet.TYPE_SCROLL_SENSITIVE
	 */
	public int getType() {
		return type;
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a java.net.URL object in the Java
	 * programming language.
	 *
	 * @param columnIndex the index of the column 1 is the first,
	 *                    2 is the second,...
	 * @return the column value as a java.net.URL object; if the value
	 *         is SQL NULL, the value returned is null in the Java
	 *         programming language
	 * @throws SQLException if a database access error occurs, or if a
	 *         URL is malformed
	 */
	public URL getURL(int columnIndex) throws SQLException {
		String url = getString(columnIndex);	
		if (url == null) return null;
		try {
			return new URL(url);
		} catch (MalformedURLException e) {
			throw new SQLException(e.getMessage(), "M1M05");
		}
	}

	/**
	 * Retrieves the value of the designated column in the current row
	 * of this ResultSet object as a java.net.URL object in the Java
	 * programming language.
	 *
	 * @param columnName the SQL name of the column
	 * @return the column value as a java.net.URL object; if the value
	 *         is SQL NULL, the value returned is null in the Java
	 *         programming language
	 * @throws SQLException if a database access error occurs, or if a
	 *         URL is malformed
	 */
	public URL getURL(String columnName) throws SQLException {
		return getURL(findColumn(columnName));
	}

	/**
	 * Retrieves the first warning reported by calls on this ResultSet object.
	 * If there is more than one warning, subsequent warnings will be chained to
	 * the first one and can be retrieved by calling the method
	 * SQLWarning.getNextWarning on the warning that was retrieved previously.
	 * <br /><br />
	 * This method may not be called on a closed result set; doing so will cause
	 * an SQLException to be thrown.
	 * <br /><br />
	 * Note: Subsequent warnings will be chained to this SQLWarning.
	 *
	 * @return the first SQLWarning object or null if there are none
	 * @throws SQLException if a database access error occurs or this method is
	 *         called on a closed connection
	 */
	public SQLWarning getWarnings() throws SQLException {
		if (header.isClosed())
			throw new SQLException("Cannot call on closed ResultSet", "M1M20");

		// if there are no warnings, this will be null, which fits with the
		// specification.
		return warnings;
	}

	/**
	 * Retrieves whether the cursor is after the last row in this ResultSet
	 * object.
	 *
	 * @return true if the cursor is after the last row; false if the cursor is
	 *         at any other position or the result set contains no rows
	 */
	public boolean isAfterLast() {
		return curRow == tupleCount + 1;
	}

	/**
	 * Retrieves whether the cursor is before the first row in this ResultSet
	 * object.
	 *
	 * @return true if the cursor is before the first row; false if the cursor
	 *         is at any other position or the result set contains no rows
	 */
	public boolean isBeforeFirst() {
		return curRow == 0;
	}

	/**
	 * Retrieves whether this ResultSet object has been closed. A
	 * ResultSet is closed if the method close has been called on it, or
	 * if it is automatically closed.
	 *
	 * @return true if this ResultSet object is closed; false if it is
	 *         still open
	 */
	public boolean isClosed() {
		return header.isClosed();
	}

	/**
	 * Retrieves whether the cursor is on the first row of this ResultSet
	 * object.
	 *
	 * @return true if the cursor is on the first row; false otherwise
	 */
	public boolean isFirst() {
		return curRow == 1;
	}

	/**
	 * Retrieves whether the cursor is on the last row of this ResultSet object.
	 *
	 * @return true if the cursor is on the last row; false otherwise
	 */
	public boolean isLast() {
		return curRow == tupleCount;
	}

	/**
	 * Moves the cursor to the last row in this ResultSet object.
	 *
	 * @return true if the cursor is on a valid row; false if there are no rows
	 *         in the result set
	 * @throws SQLException if a database access error occurs or the result set
	 *         type is TYPE_FORWARD_ONLY
	 */
	public boolean last() throws SQLException {
		return absolute(-1);
	}

	/**
	 * Moves the cursor down one row from its current position. A ResultSet
	 * cursor is initially positioned before the first row; the first call to
	 * the method next makes the first row the current row; the second call
	 * makes the second row the current row, and so on.
	 * <br /><br />
	 * If an input stream is open for the current row, a call to the method
	 * next will implicitly close it. A ResultSet object's warning chain is
	 * cleared when a new row is read.
	 *
	 * @return true if the new current row is valid; false if there are no
	 *         more rows
	 * @throws SQLException if a database access error occurs or ResultSet is
	 *         closed
	 */
	public boolean next() throws SQLException {
		return relative(1);
	}

	/**
	 * Moves the cursor to the previous row in this ResultSet object.
	 *
	 * @return true if the cursor is on a valid row; false if it is off
	 *         the result set
	 * @throws SQLException if a database access error occurs or ResultSet is
	 *         closed or the result set type is TYPE_FORWARD_ONLY
	 */
	public boolean previous() throws SQLException {
		return relative(-1);
	}

	/**
	 * Moves the cursor a relative number of rows, either positive or negative.
	 * Attempting to move beyond the first/last row in the result set positions
	 * the cursor before/after the the first/last row. Calling relative(0) is
	 * valid, but does not change the cursor position.
	 * <br /><br />
	 * Note: Calling the method relative(1) is identical to calling the method
	 * next() and calling the method relative(-1) is identical to calling the
	 * method previous().
	 *
	 * @param rows an int specifying the number of rows to move from the current
	 *        row; a positive number moves the cursor forward; a negative number
	 *        moves the cursor backward
	 * @return true if the cursor is on a row; false otherwise
	 * @throws SQLException if a database access error occurs, there is no current
	 *         row, or the result set type is TYPE_FORWARD_ONLY
	 */
	public boolean relative(int rows) throws SQLException {
		return absolute(curRow + rows);
	}

	/* these methods are all related to updateable result sets, which we
	   currently do not support */
	public void cancelRowUpdates() throws SQLException { throw new SQLFeatureNotSupportedException("Method cancelRowUpdates not implemented yet, sorry!", "0A000"); }
	public void deleteRow() throws SQLException { throw new SQLFeatureNotSupportedException("Method deleteRow not implemented yet, sorry!", "0A000"); }
	public void insertRow() throws SQLException { throw new SQLFeatureNotSupportedException("Method insertRow not implemented yet, sorry!", "0A000"); }
	public void moveToCurrentRow() throws SQLException { throw new SQLFeatureNotSupportedException("Method moveToCurrentRow not implemented yet, sorry!", "0A000"); }
	public void moveToInsertRow() throws SQLException { throw new SQLFeatureNotSupportedException("Method moveToInsertRow not implemented yet, sorry!", "0A000"); }
	public void refreshRow() throws SQLException { throw new SQLFeatureNotSupportedException("Method refreshRow not implemented yet, sorry!", "0A000"); }
	public boolean rowDeleted() throws SQLException { throw new SQLFeatureNotSupportedException("Method rowDeleted not implemented yet, sorry!", "0A000"); }
	public boolean rowInserted() throws SQLException { throw new SQLFeatureNotSupportedException("Method rowInserted not implemented yet, sorry!", "0A000"); }
	public boolean rowUpdated() throws SQLException { throw new SQLFeatureNotSupportedException("Method rowUpdated not implemented yet, sorry!", "0A000"); }
	public void setFetchDirection(int direction) throws SQLException { throw new SQLFeatureNotSupportedException("Method setFetchDirection not implemented yet, sorry!", "0A000"); }
	public void setFetchSize(int rows) throws SQLException { throw new SQLFeatureNotSupportedException("Method setFetchSize not implemented yet, sorry!", "0A000"); }
	public void updateArray(int columnIndex, Array x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateArray not implemented yet, sorry!", "0A000"); }
	public void updateArray(String columnName, Array x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateArray not implemented yet, sorry!", "0A000"); }
	public void updateAsciiStream(int columnIndex, InputStream xh) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateAsciiStream not implemented yet, sorry!", "0A000"); }
	public void updateAsciiStream(int columnIndex, InputStream x, int length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateAsciiStream not implemented yet, sorry!", "0A000"); }
	public void updateAsciiStream(int columnIndex, InputStream x, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateAsciiStream not implemented yet, sorry!", "0A000"); }
	public void updateAsciiStream(String columnName, InputStream x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateAsciiStream not implemented yet, sorry!", "0A000"); }
	public void updateAsciiStream(String columnName, InputStream x, int length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateAsciiStream not implemented yet, sorry!", "0A000"); }
	public void updateAsciiStream(String columnName, InputStream x, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateAsciiStream not implemented yet, sorry!", "0A000"); }
	public void updateBigDecimal(int columnIndex, BigDecimal x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBigDecimal not implemented yet, sorry!", "0A000"); }
	public void updateBigDecimal(String columnName, BigDecimal x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBigDecimal not implemented yet, sorry!", "0A000"); }
	public void updateBinaryStream(int columnIndex, InputStream x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBinaryStream not implemented yet, sorry!", "0A000"); }
	public void updateBinaryStream(int columnIndex, InputStream x, int length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBinaryStream not implemented yet, sorry!", "0A000"); }
	public void updateBinaryStream(int columnIndex, InputStream x, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBinaryStream not implemented yet, sorry!", "0A000"); }
	public void updateBinaryStream(String columnName, InputStream x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBinaryStream not implemented yet, sorry!", "0A000"); }
	public void updateBinaryStream(String columnName, InputStream x, int length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBinaryStream not implemented yet, sorry!", "0A000"); }
	public void updateBinaryStream(String columnName, InputStream x, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBinaryStream not implemented yet, sorry!", "0A000"); }
	public void updateBlob(int columnIndex, Blob x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBlob not implemented yet, sorry!", "0A000"); }
	public void updateBlob(int columnIndex, InputStream s) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBlob not implemented yet, sorry!", "0A000"); }
	public void updateBlob(int columnIndex, InputStream s, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBlob not implemented yet, sorry!", "0A000"); }
	public void updateBlob(String columnName, Blob x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBlob not implemented yet, sorry!", "0A000"); }
	public void updateBlob(String columnName, InputStream s) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBlob not implemented yet, sorry!", "0A000"); }
	public void updateBlob(String columnName, InputStream s, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBlob not implemented yet, sorry!", "0A000"); }
	public void updateBoolean(int columnIndex, boolean x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBoolean not implemented yet, sorry!", "0A000"); }
	public void updateBoolean(String columnName, boolean x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBoolean not implemented yet, sorry!", "0A000"); }
	public void updateByte(int columnIndex, byte x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateByte not implemented yet, sorry!", "0A000"); }
	public void updateByte(String columnName, byte x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateByte not implemented yet, sorry!", "0A000"); }
	public void updateBytes(int columnIndex, byte[] x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBytes not implemented yet, sorry!", "0A000"); }
	public void updateBytes(String columnName, byte[] x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateBytes not implemented yet, sorry!", "0A000"); }
	public void updateCharacterStream(int columnIndex, Reader x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateCharacterStream(int columnIndex, Reader x, int length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateCharacterStream(int columnIndex, Reader x, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateCharacterStream(String columnName, Reader reader) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateCharacterStream(String columnName, Reader reader, int length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateCharacterStream(String columnName, Reader reader, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateNCharacterStream(int columnIndex, Reader x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateNCharacterStream(int columnIndex, Reader x, int length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateNCharacterStream(int columnIndex, Reader x, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateNCharacterStream(String columnName, Reader reader) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateNCharacterStream(String columnName, Reader reader, int length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateNCharacterStream(String columnName, Reader reader, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNCharacterStream not implemented yet, sorry!", "0A000"); }
	public void updateClob(int columnIndex, Clob x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateClob not implemented yet, sorry!", "0A000"); }
	public void updateClob(int columnIndex, Reader r) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateClob not implemented yet, sorry!", "0A000"); }
	public void updateClob(int columnIndex, Reader r, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateClob not implemented yet, sorry!", "0A000"); }
	public void updateClob(String columnName, Clob x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateClob not implemented yet, sorry!", "0A000"); }
	public void updateClob(String columnName, Reader r) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateClob not implemented yet, sorry!", "0A000"); }
	public void updateClob(String columnName, Reader r, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateClob not implemented yet, sorry!", "0A000"); }
	public void updateNClob(int columnIndex, NClob x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNClob not implemented yet, sorry!", "0A000"); }
	public void updateNClob(int columnIndex, Reader r) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNClob not implemented yet, sorry!", "0A000"); }
	public void updateNClob(int columnIndex, Reader r, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNClob not implemented yet, sorry!", "0A000"); }
	public void updateNClob(String columnName, NClob x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNClob not implemented yet, sorry!", "0A000"); }
	public void updateNClob(String columnName, Reader r) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNClob not implemented yet, sorry!", "0A000"); }
	public void updateNClob(String columnName, Reader r, long length) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNClob not implemented yet, sorry!", "0A000"); }
	public void updateDate(int columnIndex, java.sql.Date x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateDate not implemented yet, sorry!", "0A000"); }
	public void updateDate(String columnName, java.sql.Date x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateDate not implemented yet, sorry!", "0A000"); }
	public void updateDouble(int columnIndex, double x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateDouble not implemented yet, sorry!", "0A000"); }
	public void updateDouble(String columnName, double x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateDouble not implemented yet, sorry!", "0A000"); }
	public void updateFloat(int columnIndex, float x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateFloat not implemented yet, sorry!", "0A000"); }
	public void updateFloat(String columnName, float x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateFloat not implemented yet, sorry!", "0A000"); }
	public void updateInt(int columnIndex, int x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateInt not implemented yet, sorry!", "0A000"); }
	public void updateInt(String columnName, int x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateInt not implemented yet, sorry!", "0A000"); }
	public void updateLong(int columnIndex, long x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateLong not implemented yet, sorry!", "0A000"); }
	public void updateLong(String columnName, long x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateLong not implemented yet, sorry!", "0A000"); }
	public void updateNull(int columnIndex) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNull not implemented yet, sorry!", "0A000"); }
	public void updateNull(String columnName) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNull not implemented yet, sorry!", "0A000"); }
	public void updateObject(int columnIndex, Object x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateObject not implemented yet, sorry!", "0A000"); }
	public void updateObject(int columnIndex, Object x, int scale) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateObject not implemented yet, sorry!", "0A000"); }
	public void updateObject(String columnName, Object x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateObject not implemented yet, sorry!", "0A000"); }
	public void updateObject(String columnName, Object x, int scale) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateObject not implemented yet, sorry!", "0A000"); }
	public void updateRef(int columnIndex, Ref x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateRef not implemented yet, sorry!", "0A000"); }
	public void updateRef(String columnName, Ref x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateRef not implemented yet, sorry!", "0A000"); }
	public void updateRow() throws SQLException { throw new SQLFeatureNotSupportedException("Method updateRow not implemented yet, sorry!", "0A000"); }
	public void updateRowId(int columnIndex, RowId x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateRowId not implemented yet, sorry", "0A000"); }
	public void updateRowId(String columnLabel, RowId x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateRowId not implemented yet, sorry", "0A000"); }
	public void updateShort(int columnIndex, short x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateShort not implemented yet, sorry!", "0A000"); }
	public void updateShort(String columnName, short x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateShort not implemented yet, sorry!", "0A000"); }
	public void updateString(int columnIndex, String x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateString not implemented yet, sorry!", "0A000"); }
	public void updateString(String columnName, String x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateString not implemented yet, sorry!", "0A000"); }
	public void updateNString(int columnIndex, String x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNString not implemented yet, sorry!", "0A000"); }
	public void updateNString(String columnName, String x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateNString not implemented yet, sorry!", "0A000"); }
	public void updateSQLXML(String columnName, SQLXML x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateSQLXML not implemented yet, sorry!", "0A000"); }
	public void updateSQLXML(int columnIndex, SQLXML x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateSQLXML not implemented yet, sorry!", "0A000"); }
	public void updateTime(int columnIndex, Time x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateTime not implemented yet, sorry!", "0A000"); }
	public void updateTime(String columnName, Time x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateTime not implemented yet, sorry!", "0A000"); }
	public void updateTimestamp(int columnIndex, Timestamp x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateTimestamp not implemented yet, sorry!", "0A000"); }
	public void updateTimestamp(String columnName, Timestamp x) throws SQLException { throw new SQLFeatureNotSupportedException("Method updateTimestamp not implemented yet, sorry!", "0A000"); }

	// Chapter 14.2.3.3 Sun JDBC 3.0 Specification
	/**
	 * Reports whether the last column read had a value of SQL NULL. Note that
	 * you must first call one of the getter methods on a column to try to read
	 * its value and then call the method wasNull to see if the value read was
	 * SQL NULL.
	 *
	 * @return true if the last column value read was SQL NULL and false
	 *          otherwise
	 */
	public boolean wasNull() {
		return lastColumnRead != -1 ? tlp.values[lastColumnRead] == null : false;
	}

	//== end methods of interface ResultSet

	/**
	 * Adds a warning to the pile of warnings this ResultSet object has. If
	 * there were no warnings (or clearWarnings was called) this warning will
	 * be the first, otherwise this warning will get appended to the current
	 * warning.
	 *
	 * @param reason the warning message
	 */
	private void addWarning(String reason, String sqlstate) {
		if (warnings == null) {
			warnings = new SQLWarning(reason, sqlstate);
		} else {
			warnings.setNextWarning(new SQLWarning(reason, sqlstate));
		}
	}

	/**
	 * Wrapper function to map a BLOB as BINARY.  Some applications
	 * require a BINARY type that maps into a byte array, that
	 * MonetDB/SQL lacks.  With the treat_blob_as_binary flag from the
	 * MonetDriver, we can "fake" the BINARY type, as replacement of the
	 * BLOB type.  This functions calls the getJavaType functions of the
	 * MonetDriver, but changes BLOB to BINARY if necessary afterwards.
	 *
	 * @param sqltype the string sqltype that the server knows
	 * @return a java.sql.Types constant matching sqltype, with BLOB
	 *         mapped to BINARY if requested.
	 */
	private int getJavaType(String sqltype) throws SQLException {
		int type = MonetDriver.getJavaType(sqltype);
		if (type == Types.BLOB) {
			if (statement != null && ((MonetConnection)statement.getConnection()).getBlobAsBinary())
				type = Types.BINARY;
		}
		return type;
	}
}
