/*
 * The contents of this file are subject to the MonetDB Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.monetdb.org/Legal/MonetDBLicense
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is the MonetDB Database System.
 *
 * The Initial Developer of the Original Code is CWI.
 * Portions created by CWI are Copyright (C) 1997-July 2008 CWI.
 * Copyright August 2008-2015 MonetDB B.V.
 * All Rights Reserved.
 */

/*
 * This code was created by Peter Harvey (mostly during Christmas 98/99).
 * This code is LGPL. Please ensure that this message remains in future
 * distributions and uses of this code (thats about all I get out of it).
 * - Peter Harvey pharvey@codebydesign.com
 *
 * This file has been modified for the MonetDB project.  See the file
 * Copyright in this directory for more information.
 */

/**********************************************************************
 * SQLGetInfo()
 * CLI Compliance: ISO 92
 *
 * Author: Martin van Dinther, Sjoerd Mullender
 * Date  : 30 Aug 2002
 *
 **********************************************************************/

#include "ODBCGlobal.h"
#include "ODBCDbc.h"
#include "ODBCUtil.h"


static SQLRETURN
SQLGetInfo_(ODBCDbc *dbc,
	    SQLUSMALLINT InfoType,
	    SQLPOINTER InfoValuePtr,
	    SQLSMALLINT BufferLength,
	    SQLSMALLINT *StringLengthPtr)
{
	int nValue = 0;
	char buf[64];
	const char *sValue = NULL;	/* iff non-NULL, return string value */
	int len = sizeof(SQLUINTEGER);	/* most common size to return */

	/* For some info types an active connection is needed */
	if (!dbc->Connected &&
	    (InfoType == SQL_DATA_SOURCE_NAME ||
	     InfoType == SQL_SERVER_NAME ||
	     InfoType == SQL_DATABASE_NAME ||
	     InfoType == SQL_USER_NAME)) {
		/* Connection does not exist */
		addDbcError(dbc, "08003", NULL, 0);
		return SQL_ERROR;
	}

	switch (InfoType) {
	case SQL_ACCESSIBLE_PROCEDURES:
		sValue = "Y";	/* "N" */
		break;
	case SQL_ACCESSIBLE_TABLES:
		sValue = "N";	/* "Y" */
		break;
	case SQL_ACTIVE_ENVIRONMENTS:
		nValue = 0;	/* 0 = no limit */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_AGGREGATE_FUNCTIONS:
		nValue = SQL_AF_ALL |
			SQL_AF_AVG |
			SQL_AF_COUNT |
			SQL_AF_DISTINCT |
			SQL_AF_MAX |
			SQL_AF_MIN |
			SQL_AF_SUM;
		break;
	case SQL_ALTER_DOMAIN:
		/* SQL_AD_ADD_CONSTRAINT_DEFERRABLE |
		 * SQL_AD_ADD_CONSTRAINT_INITIALLY_DEFERRED |
		 * SQL_AD_ADD_CONSTRAINT_INITIALLY_IMMEDIATE |
		 * SQL_AD_ADD_CONSTRAINT_NON_DEFERRABLE |
		 * SQL_AD_ADD_DOMAIN_CONSTRAINT |
		 * SQL_AD_ADD_DOMAIN_DEFAULT |
		 * SQL_AD_CONSTRAINT_NAME_DEFINITION |
		 * SQL_AD_DROP_DOMAIN_CONSTRAINT |
		 * SQL_AD_DROP_DOMAIN_DEFAULT */
		break;
	case SQL_ALTER_TABLE:
		nValue = SQL_AT_ADD_COLUMN | /* ODBC 2.0, deprecated value */
			SQL_AT_ADD_COLUMN_DEFAULT |
			SQL_AT_ADD_COLUMN_SINGLE |
			SQL_AT_ADD_CONSTRAINT |
			SQL_AT_ADD_TABLE_CONSTRAINT |
			SQL_AT_CONSTRAINT_NAME_DEFINITION |
			SQL_AT_DROP_COLUMN | /* ODBC 2.0, deprecated value */
			SQL_AT_DROP_COLUMN_CASCADE |
			SQL_AT_DROP_COLUMN_DEFAULT |
			SQL_AT_DROP_COLUMN_RESTRICT |
			SQL_AT_DROP_TABLE_CONSTRAINT_CASCADE |
			SQL_AT_DROP_TABLE_CONSTRAINT_RESTRICT |
			SQL_AT_SET_COLUMN_DEFAULT;
		/* SQL_AT_ADD_COLUMN_COLLATION |
		 * SQL_AT_CONSTRAINT_DEFERRABLE |
		 * SQL_AT_CONSTRAINT_INITIALLY_DEFERRED |
		 * SQL_AT_CONSTRAINT_INITIALLY_IMMEDIATE |
		 * SQL_AT_CONSTRAINT_NON_DEFERRABLE */
		break;
#ifdef SQL_ASYNC_DBC_FUNCTIONS
	case SQL_ASYNC_DBC_FUNCTIONS:
		nValue = SQL_ASYNC_DBC_NOT_CAPABLE;
		/* SQL_ASYNC_DBC_CAPABLE */
		break;
#endif
	case SQL_ASYNC_MODE:
		nValue = SQL_AM_NONE;
		/* SQL_AM_CONNECTION, SQL_AM_STATEMENT */
		break;
#ifdef SQL_ASYNC_NOTIFICATION
	case SQL_ASYNC_NOTIFICATION:
		nValue = SQL_ASYNC_NOTIFICATION_NOT_CAPABLE;
		/* SQL_ASYNC_NOTIFICATION_CAPABLE */
		break;
#endif
	case SQL_BATCH_ROW_COUNT:
		nValue = SQL_BRC_EXPLICIT;
		/* SQL_BRC_PROCEDURES | SQL_BRC_ROLLED_UP */
		break;
	case SQL_BATCH_SUPPORT:
		nValue = SQL_BS_ROW_COUNT_EXPLICIT | SQL_BS_SELECT_EXPLICIT;
		/* SQL_BS_ROW_COUNT_PROC |
		 * SQL_BS_SELECT_PROC */
		break;
	case SQL_BOOKMARK_PERSISTENCE:
		/* SQL_BP_CLOSE |
		 * SQL_BP_DELETE |
		 * SQL_BP_DROP |
		 * SQL_BP_OTHER_HSTMT |
		 * SQL_BP_TRANSACTION |
		 * SQL_BP_UPDATE */
		break;
	case SQL_CATALOG_LOCATION:
		/* SQL_CL_END, SQL_CL_START */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_CATALOG_NAME:
		sValue = "N";	/* "Y" */
		break;
	case SQL_CATALOG_NAME_SEPARATOR:
	case SQL_CATALOG_TERM:
		sValue = "";
		break;
	case SQL_CATALOG_USAGE:
		/* SQL_CU_DML_STATEMENTS |
		 * SQL_CU_INDEX_DEFINITION |
		 * SQL_CU_PRIVILEGE_DEFINITION |
		 * SQL_CU_PROCEDURE_INVOCATION |
		 * SQL_CU_TABLE_DEFINITION */
		break;
	case SQL_COLLATION_SEQ:
		sValue = "UTF-8";
		break;
	case SQL_COLUMN_ALIAS:
		sValue = "Y";	/* "N" */
		break;
	case SQL_CONCAT_NULL_BEHAVIOR:
		nValue = SQL_CB_NULL;
		/* SQL_CB_NON_NULL */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_CONVERT_TINYINT:
	case SQL_CONVERT_SMALLINT:
	case SQL_CONVERT_INTEGER:
	case SQL_CONVERT_BIGINT:
		nValue = SQL_CVT_BIGINT |
			SQL_CVT_BIT |
			SQL_CVT_CHAR |
			SQL_CVT_DECIMAL |
			SQL_CVT_DOUBLE |
			SQL_CVT_FLOAT |
			SQL_CVT_INTEGER |
			SQL_CVT_INTERVAL_DAY_TIME |
			SQL_CVT_INTERVAL_YEAR_MONTH |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_NUMERIC |
			SQL_CVT_REAL |
			SQL_CVT_SMALLINT |
			SQL_CVT_TINYINT |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_BINARY:
	case SQL_CONVERT_LONGVARBINARY:
	case SQL_CONVERT_VARBINARY:
		nValue = SQL_CVT_BINARY |
			SQL_CVT_CHAR |
			SQL_CVT_LONGVARBINARY |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_VARBINARY |
			SQL_CVT_VARCHAR;
		/* SQL_CVT_GUID */
		break;
	case SQL_CONVERT_BIT:
		nValue = SQL_CVT_BIGINT |
			SQL_CVT_BIT |
			SQL_CVT_CHAR |
			SQL_CVT_INTEGER |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_SMALLINT |
			SQL_CVT_TINYINT |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_CHAR:
	case SQL_CONVERT_VARCHAR:
	case SQL_CONVERT_LONGVARCHAR:
		nValue = SQL_CVT_BIGINT |
			SQL_CVT_BINARY |
			SQL_CVT_BIT |
			SQL_CVT_CHAR |
			SQL_CVT_DATE |
			SQL_CVT_DECIMAL |
			SQL_CVT_DOUBLE |
			SQL_CVT_FLOAT |
			SQL_CVT_GUID |
			SQL_CVT_INTEGER |
			SQL_CVT_INTERVAL_DAY_TIME |
			SQL_CVT_INTERVAL_YEAR_MONTH |
			SQL_CVT_LONGVARBINARY |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_NUMERIC |
			SQL_CVT_REAL |
			SQL_CVT_SMALLINT |
			SQL_CVT_TIME |
			SQL_CVT_TIMESTAMP |
			SQL_CVT_TINYINT |
			SQL_CVT_VARBINARY |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_DATE:
		nValue = SQL_CVT_CHAR |
			SQL_CVT_DATE |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_TIMESTAMP |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_DECIMAL:
	case SQL_CONVERT_NUMERIC:
		nValue = SQL_CVT_BIGINT |
			SQL_CVT_CHAR |
			SQL_CVT_DECIMAL |
			SQL_CVT_DOUBLE |
			SQL_CVT_FLOAT |
			SQL_CVT_INTEGER |
			SQL_CVT_INTERVAL_DAY_TIME |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_NUMERIC |
			SQL_CVT_REAL |
			SQL_CVT_SMALLINT |
			SQL_CVT_TINYINT |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_DOUBLE:
	case SQL_CONVERT_REAL:
	case SQL_CONVERT_FLOAT:
		nValue = SQL_CVT_BIGINT |
			SQL_CVT_CHAR |
			SQL_CVT_DECIMAL |
			SQL_CVT_DOUBLE |
			SQL_CVT_FLOAT |
			SQL_CVT_INTEGER |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_NUMERIC |
			SQL_CVT_REAL |
			SQL_CVT_SMALLINT |
			SQL_CVT_TINYINT |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_INTERVAL_DAY_TIME:
		nValue = SQL_CVT_BIGINT |
			SQL_CVT_CHAR |
			SQL_CVT_INTEGER |
			SQL_CVT_INTERVAL_DAY_TIME |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_SMALLINT |
			SQL_CVT_TIME |
			SQL_CVT_TINYINT |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_INTERVAL_YEAR_MONTH:
		nValue = SQL_CVT_BIGINT |
			SQL_CVT_CHAR |
			SQL_CVT_INTEGER |
			SQL_CVT_INTERVAL_YEAR_MONTH |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_SMALLINT |
			SQL_CVT_TINYINT |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_TIME:
		nValue = SQL_CVT_CHAR |
			SQL_CVT_INTERVAL_DAY_TIME |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_TIME |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_TIMESTAMP:
		nValue = SQL_CVT_CHAR |
			SQL_CVT_DATE |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_TIME |
			SQL_CVT_TIMESTAMP |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_GUID:
		nValue =SQL_CVT_CHAR |
			SQL_CVT_GUID |
			SQL_CVT_LONGVARCHAR |
			SQL_CVT_VARCHAR;
		break;
	case SQL_CONVERT_FUNCTIONS:
		nValue = SQL_FN_CVT_CAST | SQL_FN_CVT_CONVERT;
		break;
	case SQL_CORRELATION_NAME:
		nValue = SQL_CN_ANY;
		/* SQL_CN_DIFFERENT, SQL_CN_NONE */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_CREATE_ASSERTION:
		/* SQL_CA_CREATE_ASSERTION |
		 * SQL_CA_CONSTRAINT_DEFERRABLE |
		 * SQL_CA_CONSTRAINT_INITIALLY_DEFERRED |
		 * SQL_CA_CONSTRAINT_INITIALLY_IMMEDIATE |
		 * SQL_CA_CONSTRAINT_NON_DEFERRABLE */
	case SQL_CREATE_CHARACTER_SET:
		/* SQL_CCS_CREATE_CHARACTER_SET |
		 * SQL_CCS_COLLATE_CLAUSE |
		 * SQL_CCS_LIMITED_COLLATION */
	case SQL_CREATE_COLLATION:
		/* SQL_CCOL_CREATE_COLLATION */
	case SQL_CREATE_DOMAIN:
		/* SQL_CDO_CREATE_DOMAIN |
		 * SQL_CDO_CONSTRAINT_NAME_DEFINITION |
		 * SQL_CDO_DEFAULT |
		 * SQL_CDO_CONSTRAINT |
		 * SQL_CDO_COLLATION |
		 * SQL_CDO_CONSTRAINT_DEFERRABLE |
		 * SQL_CDO_CONSTRAINT_INITIALLY_DEFERRED |
		 * SQL_CDO_CONSTRAINT_INITIALLY_IMMEDIATE |
		 * SQL_CDO_CONSTRAINT_NON_DEFERRABLE */
	case SQL_CREATE_TRANSLATION:
		/* SQL_CTR_CREATE_TRANSLATION */
		break;
	case SQL_CREATE_SCHEMA:
		nValue = SQL_CS_CREATE_SCHEMA | SQL_CS_AUTHORIZATION;
		/* SQL_CS_DEFAULT_CHARACTER_SET */
		break;
	case SQL_CREATE_TABLE:
		nValue = SQL_CT_COLUMN_CONSTRAINT |
			SQL_CT_COLUMN_DEFAULT |
			SQL_CT_COMMIT_DELETE |
			SQL_CT_COMMIT_PRESERVE |
			SQL_CT_CONSTRAINT_NAME_DEFINITION |
			SQL_CT_CREATE_TABLE |
			SQL_CT_GLOBAL_TEMPORARY |
			SQL_CT_LOCAL_TEMPORARY |
			SQL_CT_TABLE_CONSTRAINT;
		/* SQL_CT_COLUMN_COLLATION |
		 * SQL_CT_CONSTRAINT_DEFERRABLE |
		 * SQL_CT_CONSTRAINT_INITIALLY_DEFERRED |
		 * SQL_CT_CONSTRAINT_INITIALLY_IMMEDIATE |
		 * SQL_CT_CONSTRAINT_NON_DEFERRABLE */
		break;
	case SQL_CREATE_VIEW:
		nValue = SQL_CV_CREATE_VIEW | SQL_CV_CHECK_OPTION;
		/* SQL_CV_CASCADE | SQL_CV_LOCAL */
		break;
	case SQL_CURSOR_COMMIT_BEHAVIOR:
	case SQL_CURSOR_ROLLBACK_BEHAVIOR:
		nValue = SQL_CB_DELETE;
		/* SQL_CB_CLOSE, SQL_CB_DELETE, SQL_CB_PRESERVE */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_CURSOR_SENSITIVITY:
		nValue = SQL_INSENSITIVE;
		/* SQL_SENSITIVE, SQL_UNSPECIFIED */
		break;
	case SQL_DATA_SOURCE_NAME:
		sValue = dbc->dsn ? dbc->dsn : "";
		break;
	case SQL_DATA_SOURCE_READ_ONLY:
		sValue = "N";	/* "Y" */
		break;
	case SQL_DATABASE_NAME:
		sValue = dbc->dbname ? dbc->dbname : "";
		break;
	case SQL_DATETIME_LITERALS:
		nValue = SQL_DL_SQL92_DATE |
			SQL_DL_SQL92_TIME |
			SQL_DL_SQL92_TIMESTAMP |
			SQL_DL_SQL92_INTERVAL_YEAR |
			SQL_DL_SQL92_INTERVAL_MONTH |
			SQL_DL_SQL92_INTERVAL_DAY |
			SQL_DL_SQL92_INTERVAL_HOUR |
			SQL_DL_SQL92_INTERVAL_MINUTE |
			SQL_DL_SQL92_INTERVAL_SECOND |
			SQL_DL_SQL92_INTERVAL_YEAR_TO_MONTH |
			SQL_DL_SQL92_INTERVAL_DAY_TO_HOUR |
			SQL_DL_SQL92_INTERVAL_DAY_TO_MINUTE |
			SQL_DL_SQL92_INTERVAL_DAY_TO_SECOND |
			SQL_DL_SQL92_INTERVAL_HOUR_TO_MINUTE |
			SQL_DL_SQL92_INTERVAL_HOUR_TO_SECOND |
			SQL_DL_SQL92_INTERVAL_MINUTE_TO_SECOND;
		len = sizeof(SQLINTEGER);
		break;
	case SQL_DBMS_NAME:
		sValue = PACKAGE_NAME;
		break;
	case SQL_DBMS_VER:
		snprintf(buf, sizeof(buf), "%02d.%02d.%04d",
			 dbc->major, dbc->minor, dbc->patch);
		sValue = buf;
		break;
	case SQL_DDL_INDEX:
		nValue = SQL_DI_CREATE_INDEX | SQL_DI_DROP_INDEX;
		break;
	case SQL_DEFAULT_TXN_ISOLATION:
		nValue = SQL_TXN_SERIALIZABLE;
		break;
	case SQL_DESCRIBE_PARAMETER:
		sValue = "N";	/* "Y" */
		break;
#ifdef SQL_DRIVER_AWARE_POOLING_SUPPORTED
	case SQL_DRIVER_AWARE_POOLING_SUPPORTED:
		nValue = SQL_DRIVER_AWARE_POOLING_NOT_CAPABLE;
		/* SQL_DRIVER_AWARE_POOLING_CAPABLE */
		break;
#endif
	case SQL_DRIVER_NAME:
		sValue = MONETDB_DRIVER_NAME;
		break;
	case SQL_DRIVER_ODBC_VER:
		sValue = MONETDB_ODBC_VER;
		break;
	case SQL_DRIVER_VER: {
		int maj = 0, min = 0, pat = 0;
		sscanf(PACKAGE_VERSION, "%d.%d.%d", &maj, &min, &pat);
		snprintf(buf, sizeof(buf), "%02d.%02d.%04d %s", maj, min, pat,
			 MONETDB_RELEASE);
		sValue = buf;
		break;
	}
	case SQL_DROP_ASSERTION:
		/* SQL_DA_DROP_ASSERION */
		break;
	case SQL_DROP_CHARACTER_SET:
		/* SQL_DCS_DROP_CHARACTER_SET */
		break;
	case SQL_DROP_COLLATION:
		/* SQL_DC_DROP_COLLATION */
		break;
	case SQL_DROP_DOMAIN:
		/* SQL_DD_DROP_DOMAIN | SQL_DD_CASCADE | SQL_DD_RESTRICT */
		break;
	case SQL_DROP_SCHEMA:
		nValue = SQL_DS_DROP_SCHEMA | SQL_DS_CASCADE | SQL_DS_RESTRICT;
		break;
	case SQL_DROP_TABLE:
		nValue = SQL_DT_DROP_TABLE | SQL_DT_CASCADE | SQL_DT_RESTRICT;
		break;
	case SQL_DROP_TRANSLATION:
		/* SQL_DTR_DROP_TRANSLATION */
		break;
	case SQL_DROP_VIEW:
		nValue = SQL_DV_DROP_VIEW | SQL_DV_CASCADE | SQL_DV_RESTRICT;
		break;
	case SQL_DYNAMIC_CURSOR_ATTRIBUTES1:
		nValue = SQL_CA1_ABSOLUTE | SQL_CA1_NEXT | SQL_CA1_RELATIVE;
		/* SQL_CA1_BOOKMARK |
		 * SQL_CA1_BULK_ADD |
		 * SQL_CA1_BULK_DELETE_BY_BOOKMARK |
		 * SQL_CA1_BULK_FETCH_BY_BOOKMARK |
		 * SQL_CA1_BULK_UPDATE_BY_BOOKMARK |
		 * SQL_CA1_LOCK_EXCLUSIVE |
		 * SQL_CA1_LOCK_NO_CHANGE |
		 * SQL_CA1_LOCK_UNLOCK |
		 * SQL_CA1_POS_DELETE |
		 * SQL_CA1_POSITIONED_DELETE |
		 * SQL_CA1_POSITIONED_UPDATE |
		 * SQL_CA1_POS_POSITION |
		 * SQL_CA1_POS_REFRESH |
		 * SQL_CA1_POS_UPDATE |
		 * SQL_CA1_SELECT_FOR_UPDATE */
		break;
	case SQL_DYNAMIC_CURSOR_ATTRIBUTES2:
		/* SQL_CA2_CRC_APPROXIMATE |
		 * SQL_CA2_CRC_EXACT |
		 * SQL_CA2_LOCK_CONCURRENCY |
		 * SQL_CA2_MAX_ROWS_AFFECTS_ALL |
		 * SQL_CA2_MAX_ROWS_CATALOG |
		 * SQL_CA2_MAX_ROWS_DELETE |
		 * SQL_CA2_MAX_ROWS_INSERT |
		 * SQL_CA2_MAX_ROWS_SELECT |
		 * SQL_CA2_MAX_ROWS_UPDATE |
		 * SQL_CA2_OPT_ROWVER_CONCURRENCY |
		 * SQL_CA2_OPT_VALUES_CONCURRENCY |
		 * SQL_CA2_READ_ONLY_CONCURRENCY |
		 * SQL_CA2_SENSITIVITY_ADDITIONS |
		 * SQL_CA2_SENSITIVITY_DELETIONS |
		 * SQL_CA2_SENSITIVITY_UPDATES |
		 * SQL_CA2_SIMULATE_NON_UNIQUE |
		 * SQL_CA2_SIMULATE_TRY_UNIQUE |
		 * SQL_CA2_SIMULATE_UNIQUE */
		break;
	case SQL_EXPRESSIONS_IN_ORDERBY:
		sValue = "Y";	/* "N" */
		break;
	case SQL_FILE_USAGE:
		nValue = SQL_FILE_NOT_SUPPORTED;
		/* SQL_FILE_TABLE, SQL_FILE_CATALOG */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_FORWARD_ONLY_CURSOR_ATTRIBUTES1:
		nValue = SQL_CA1_NEXT;
		/* SQL_CA1_BULK_ADD |
		 * SQL_CA1_BULK_DELETE_BY_BOOKMARK |
		 * SQL_CA1_BULK_FETCH_BY_BOOKMARK |
		 * SQL_CA1_BULK_UPDATE_BY_BOOKMARK |
		 * SQL_CA1_LOCK_EXCLUSIVE |
		 * SQL_CA1_LOCK_NO_CHANGE |
		 * SQL_CA1_LOCK_UNLOCK |
		 * SQL_CA1_POS_DELETE |
		 * SQL_CA1_POSITIONED_DELETE |
		 * SQL_CA1_POSITIONED_UPDATE |
		 * SQL_CA1_POS_POSITION |
		 * SQL_CA1_POS_REFRESH |
		 * SQL_CA1_POS_UPDATE |
		 * SQL_CA1_SELECT_FOR_UPDATE */
		break;
	case SQL_FORWARD_ONLY_CURSOR_ATTRIBUTES2:
		/* SQL_CA2_CRC_APPROXIMATE |
		 * SQL_CA2_CRC_EXACT |
		 * SQL_CA2_LOCK_CONCURRENCY |
		 * SQL_CA2_MAX_ROWS_AFFECTS_ALL |
		 * SQL_CA2_MAX_ROWS_CATALOG |
		 * SQL_CA2_MAX_ROWS_DELETE |
		 * SQL_CA2_MAX_ROWS_INSERT |
		 * SQL_CA2_MAX_ROWS_SELECT |
		 * SQL_CA2_MAX_ROWS_UPDATE |
		 * SQL_CA2_OPT_ROWVER_CONCURRENCY |
		 * SQL_CA2_OPT_VALUES_CONCURRENCY |
		 * SQL_CA2_READ_ONLY_CONCURRENCY |
		 * SQL_CA2_SENSITIVITY_ADDITIONS |
		 * SQL_CA2_SENSITIVITY_DELETIONS |
		 * SQL_CA2_SENSITIVITY_UPDATES |
		 * SQL_CA2_SIMULATE_NON_UNIQUE |
		 * SQL_CA2_SIMULATE_TRY_UNIQUE |
		 * SQL_CA2_SIMULATE_UNIQUE */
		break;
	case SQL_GETDATA_EXTENSIONS:
		nValue = SQL_GD_ANY_COLUMN |
			SQL_GD_ANY_ORDER |
			SQL_GD_BLOCK |
			SQL_GD_BOUND;
		/* SQL_GD_OUTPUT_PARAMS */
		break;
	case SQL_GROUP_BY:
		nValue = SQL_GB_NO_RELATION;
		/* SQL_GB_COLLATE, SQL_GB_NOT_SUPPORTED,
		 * SQL_GB_BROUP_BY_EQUALS_SELECT,
		 * SQL_GB_GROUP_BY_CONTAINS_SELECT */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_IDENTIFIER_CASE:
		nValue = SQL_IC_LOWER;
		/* SQL_IC_UPPER, SQL_IC_SENSITIVE, SQL_IC_MIXED */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_IDENTIFIER_QUOTE_CHAR:
		sValue = "\"";	/* the " (double quote) */
		break;
	case SQL_INDEX_KEYWORDS:
		nValue = SQL_IK_NONE;
		/* SQL_IK_ASC | SQL_IK_DESC | SQL_IK_ALL */
		break;
	case SQL_INFO_SCHEMA_VIEWS:
		/* SQL_ISV_ASSERTIONS |
		 * SQL_ISV_CHARACTER_SETS |
		 * SQL_ISV_CHECK_CONSTRAINTS |
		 * SQL_ISV_COLLATIONS |
		 * SQL_ISV_COLUMN_DOMAIN_USAGE |
		 * SQL_ISV_COLUMN_PRIVILEGES |
		 * SQL_ISV_COLUMNS |
		 * SQL_ISV_CONSTRAINT_COLUMN_USAGE |
		 * SQL_ISV_CONSTRAINT_TABLE_USAGE |
		 * SQL_ISV_DOMAIN_CONSTRAINTS |
		 * SQL_ISV_DOMAINS |
		 * SQL_ISV_KEY_COLUMN_USAGE |
		 * SQL_ISV_REFERENTIAL_CONSTRAINTS |
		 * SQL_ISV_SCHEMATA |
		 * SQL_ISV_SQL_LANGUAGES |
		 * SQL_ISV_TABLE_CONSTRAINTS |
		 * SQL_ISV_TABLE_PRIVILEGES |
		 * SQL_ISV_TABLES |
		 * SQL_ISV_TRANSLATIONS |
		 * SQL_ISV_USAGE_PRIVILEGES |
		 * SQL_ISV_VIEW_COLUMN_USAGE |
		 * SQL_ISV_VIEWS |
		 * SQL_ISV_VIEW_TABLE_USAGE */
		break;
	case SQL_INSERT_STATEMENT:
		nValue = SQL_IS_INSERT_LITERALS |
			SQL_IS_INSERT_SEARCHED |
			SQL_IS_SELECT_INTO;
		break;
	case SQL_INTEGRITY:
		sValue = "N";	/* "Y" */
		break;
	case SQL_KEYSET_CURSOR_ATTRIBUTES1:
		/* SQL_CA1_ABSOLUTE |
		 * SQL_CA1_BOOKMARK |
		 * SQL_CA1_BULK_ADD |
		 * SQL_CA1_BULK_DELETE_BY_BOOKMARK |
		 * SQL_CA1_BULK_FETCH_BY_BOOKMARK |
		 * SQL_CA1_BULK_UPDATE_BY_BOOKMARK |
		 * SQL_CA1_LOCK_EXCLUSIVE |
		 * SQL_CA1_LOCK_NO_CHANGE |
		 * SQL_CA1_LOCK_UNLOCK |
		 * SQL_CA1_NEXT |
		 * SQL_CA1_POS_DELETE |
		 * SQL_CA1_POSITIONED_DELETE |
		 * SQL_CA1_POSITIONED_UPDATE |
		 * SQL_CA1_POS_POSITION |
		 * SQL_CA1_POS_REFRESH |
		 * SQL_CA1_POS_UPDATE |
		 * SQL_CA1_RELATIVE |
		 * SQL_CA1_SELECT_FOR_UPDATE */
		break;
	case SQL_KEYSET_CURSOR_ATTRIBUTES2:
		/* SQL_CA2_CRC_APPROXIMATE |
		 * SQL_CA2_CRC_EXACT |
		 * SQL_CA2_LOCK_CONCURRENCY |
		 * SQL_CA2_MAX_ROWS_AFFECTS_ALL |
		 * SQL_CA2_MAX_ROWS_CATALOG |
		 * SQL_CA2_MAX_ROWS_DELETE |
		 * SQL_CA2_MAX_ROWS_INSERT |
		 * SQL_CA2_MAX_ROWS_SELECT |
		 * SQL_CA2_MAX_ROWS_UPDATE |
		 * SQL_CA2_OPT_ROWVER_CONCURRENCY |
		 * SQL_CA2_OPT_VALUES_CONCURRENCY |
		 * SQL_CA2_READ_ONLY_CONCURRENCY |
		 * SQL_CA2_SENSITIVITY_ADDITIONS |
		 * SQL_CA2_SENSITIVITY_DELETIONS |
		 * SQL_CA2_SENSITIVITY_UPDATES |
		 * SQL_CA2_SIMULATE_NON_UNIQUE |
		 * SQL_CA2_SIMULATE_TRY_UNIQUE |
		 * SQL_CA2_SIMULATE_UNIQUE */
		break;
	case SQL_KEYWORDS:
		/* Returns the MonetDB keywords which are not listed
		 * as ODBC keyword in the #define SQL_ODBC_KEYWORDS in
		 * sql.h, collated from
		 * sql_scan.c:scanner_init_keywords with values
		 * removed that are in
		 * sql_parser.y:non_reserved_word */
		sValue = "ADMIN,AFTER,AGGREGATE,ALWAYS,ASYMMETRIC,ATOMIC,"
			"AUTO_INCREMENT,BEFORE,BIGINT,BIGSERIAL,BINARY,BLOB,"
			"CALL,CHAIN,CLOB,COMMITTED,COPY,CORR,CUME_DIST,"
			"CURRENT_ROLE,CYCLE,DATABASE,DELIMITERS,DENSE_RANK,"
			"DO,EACH,ELSEIF,ENCRYPTED,EVERY,EXCLUDE,FOLLOWING,"
			"FUNCTION,GENERATED,IF,ILIKE,INCREMENT,LAG,LAG,LEAD,"
			"LEAD,LIMIT,LOCALTIME,LOCALTIMESTAMP,LOCKED,MAXVALUE,"
			"MEDIAN,MEDIUMINT,MERGE,MINVALUE,NEW,NOCYCLE,"
			"NOMAXVALUE,NOMINVALUE,NOW,OFFSET,OLD,OTHERS,OVER,"
			"PARTITION,PERCENT_RANK,PLAN,PRECEDING,PROD,QUANTILE,"
			"RANGE,RANK,RECORDS,REFERENCING,REMOTE,RENAME,"
			"REPEATABLE,REPLICA,RESTART,RETURN,RETURNS,"
			"ROW_NUMBER,ROWS,SAMPLE,SAVEPOINT,SCHEMA,SEQUENCE,"
			"SERIAL,SERIALIZABLE,SIMPLE,START,STATEMENT,STDIN,"
			"STDOUT,STREAM,STRING,SYMMETRIC,TIES,TINYINT,TRIGGER,"
			"UNBOUNDED,UNCOMMITTED,UNENCRYPTED,WHILE,XMLAGG,"
			"XMLATTRIBUTES,XMLCOMMENT,XMLCONCAT,XMLDOCUMENT,"
			"XMLELEMENT,XMLFOREST,XMLNAMESPACES,XMLPARSE,XMLPI,"
			"XMLQUERY,XMLSCHEMA,XMLTEXT,XMLVALIDATE";
		break;
	case SQL_LIKE_ESCAPE_CLAUSE:
		sValue = "Y";	/* "N" */
		break;
	case SQL_MAX_ASYNC_CONCURRENT_STATEMENTS:
		break;
	case SQL_MAX_BINARY_LITERAL_LEN:
		break;
	case SQL_MAX_CATALOG_NAME_LEN:
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_MAX_CHAR_LITERAL_LEN:
		break;
	case SQL_MAX_COLUMN_NAME_LEN:
	case SQL_MAX_COLUMNS_IN_GROUP_BY:
	case SQL_MAX_COLUMNS_IN_INDEX:
	case SQL_MAX_COLUMNS_IN_ORDER_BY:
	case SQL_MAX_COLUMNS_IN_SELECT:
	case SQL_MAX_COLUMNS_IN_TABLE:
	case SQL_MAX_CONCURRENT_ACTIVITIES:
	case SQL_MAX_CURSOR_NAME_LEN:
	case SQL_MAX_DRIVER_CONNECTIONS:
	case SQL_MAX_IDENTIFIER_LEN:
	case SQL_MAX_INDEX_SIZE:
	case SQL_MAX_PROCEDURE_NAME_LEN:
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_MAX_ROW_SIZE:
		break;
	case SQL_MAX_ROW_SIZE_INCLUDES_LONG:
		sValue = "Y";	/* "N" */
		break;
	case SQL_MAX_SCHEMA_NAME_LEN:
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_MAX_STATEMENT_LEN:
		break;
	case SQL_MAX_TABLE_NAME_LEN:
	case SQL_MAX_TABLES_IN_SELECT:
	case SQL_MAX_USER_NAME_LEN:
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_MULT_RESULT_SETS:
		sValue = "Y";	/* "N" */
		break;
	case SQL_MULTIPLE_ACTIVE_TXN:
		sValue = "Y";	/* "N" */
		break;
	case SQL_NEED_LONG_DATA_LEN:
		sValue = "N";	/* "Y" */
		break;
	case SQL_NON_NULLABLE_COLUMNS:
		nValue = SQL_NNC_NON_NULL;
		/* SQL_NNC_NULL */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_NULL_COLLATION:
		nValue = SQL_NC_LOW;
		/* SQL_NC_END, SQL_NC_HIGH, SQL_NC_START */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_NUMERIC_FUNCTIONS:
		nValue = SQL_FN_NUM_ABS |
			SQL_FN_NUM_ACOS |
			SQL_FN_NUM_ASIN |
			SQL_FN_NUM_ATAN |
			SQL_FN_NUM_ATAN2 |
			SQL_FN_NUM_CEILING |
			SQL_FN_NUM_COS |
			SQL_FN_NUM_COT |
			SQL_FN_NUM_DEGREES |
			SQL_FN_NUM_EXP |
			SQL_FN_NUM_FLOOR |
			SQL_FN_NUM_LOG |
			SQL_FN_NUM_LOG10 |
			SQL_FN_NUM_MOD |
			SQL_FN_NUM_PI |
			SQL_FN_NUM_POWER |
			SQL_FN_NUM_RADIANS |
			SQL_FN_NUM_RAND |
			SQL_FN_NUM_ROUND |
			SQL_FN_NUM_SIGN |
			SQL_FN_NUM_SIN |
			SQL_FN_NUM_SQRT |
			SQL_FN_NUM_TAN |
			SQL_FN_NUM_TRUNCATE;
		break;
	case SQL_ODBC_INTERFACE_CONFORMANCE:
		nValue = SQL_OIC_CORE;
		/* SQL_OIC_LEVEL1, SQL_OIC_LEVEL2 */
		break;
	case SQL_OJ_CAPABILITIES:
		nValue = SQL_OJ_LEFT |
			SQL_OJ_RIGHT |
			SQL_OJ_FULL |
			SQL_OJ_NOT_ORDERED |
			SQL_OJ_INNER |
			SQL_OJ_ALL_COMPARISON_OPS;
		/* SQL_OJ_NESTED */
		break;
	case SQL_ORDER_BY_COLUMNS_IN_SELECT:
		sValue = "N";	/* "Y" */
		break;
	case SQL_PARAM_ARRAY_ROW_COUNTS:
		nValue = SQL_PARC_BATCH; /* ? */
		/* SQL_PARC_NO_BATCH */
		break;
	case SQL_PARAM_ARRAY_SELECTS:
		nValue = SQL_PAS_NO_SELECT;
		/* SQL_PAS_BATCH, SQL_PAS_NO_BATCH */
		break;
	case SQL_PROCEDURE_TERM:
		sValue = "procedure";
		break;
	case SQL_PROCEDURES:
		sValue = "Y";	/* "N" */
		break;
	case SQL_QUOTED_IDENTIFIER_CASE:
		nValue = SQL_IC_SENSITIVE;
		/* SQL_IC_LOWER, SQL_IC_MIXED, SQL_IC_UPPER */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_ROW_UPDATES:
		sValue = "N";	/* "Y" */
		break;
	case SQL_SCHEMA_TERM:
		sValue = "schema";
		break;
	case SQL_SCHEMA_USAGE:
		nValue = SQL_SU_DML_STATEMENTS |
			SQL_SU_PROCEDURE_INVOCATION |
			SQL_SU_TABLE_DEFINITION |
			SQL_SU_INDEX_DEFINITION |
			SQL_SU_PRIVILEGE_DEFINITION;
		break;
	case SQL_SCROLL_OPTIONS:
		nValue = SQL_SO_STATIC;
		/* SQL_SO_DYNAMIC,
		 * SQL_SO_FORWARD_ONLY,
		 * SQL_SO_KEYSET_DRIVEN,
		 * SQL_SO_MIXED */
		break;
	case SQL_SEARCH_PATTERN_ESCAPE:
		sValue = "\\";
		break;
	case SQL_SERVER_NAME:
		sValue = MONETDB_SERVER_NAME;
		break;
	case SQL_SPECIAL_CHARACTERS:
		sValue = "!$&'()*+,-./:;<=>?@[]^`{|}~";
		break;
	case SQL_SQL_CONFORMANCE:
		nValue = SQL_SC_SQL92_FULL;
		break;
	case SQL_SQL92_DATETIME_FUNCTIONS:
		nValue = SQL_SDF_CURRENT_DATE |
			SQL_SDF_CURRENT_TIME |
			SQL_SDF_CURRENT_TIMESTAMP;
		break;
	case SQL_SQL92_FOREIGN_KEY_DELETE_RULE: /* ? */
		nValue = SQL_SFKD_CASCADE |
			SQL_SFKD_NO_ACTION |
			SQL_SFKD_SET_DEFAULT |
			SQL_SFKD_SET_NULL;
		break;
	case SQL_SQL92_FOREIGN_KEY_UPDATE_RULE: /* ? */
		nValue = SQL_SFKU_CASCADE |
			SQL_SFKU_NO_ACTION |
			SQL_SFKU_SET_DEFAULT |
			SQL_SFKU_SET_NULL;
		break;
	case SQL_SQL92_GRANT:
		nValue = SQL_SG_DELETE_TABLE |
			SQL_SG_INSERT_COLUMN |
			SQL_SG_INSERT_TABLE |
			SQL_SG_REFERENCES_COLUMN |
			SQL_SG_REFERENCES_TABLE |
			SQL_SG_SELECT_TABLE |
			SQL_SG_UPDATE_COLUMN |
			SQL_SG_UPDATE_TABLE |
			SQL_SG_WITH_GRANT_OPTION;
		/* SQL_SG_USAGE_ON_CHARACTER_SET |
		 * SQL_SG_USAGE_ON_COLLATION |
		 * SQL_SG_USAGE_ON_DOMAIN |
		 * SQL_SG_USAGE_ON_TRANSLATION */
		break;
	case SQL_SQL92_NUMERIC_VALUE_FUNCTIONS:
		nValue = SQL_SNVF_CHAR_LENGTH |
			SQL_SNVF_CHARACTER_LENGTH |
			SQL_SNVF_EXTRACT |
			SQL_SNVF_OCTET_LENGTH |
			SQL_SNVF_POSITION;
		/* SQL_SNVF_BIT_LENGTH */
		break;
	case SQL_SQL92_PREDICATES: /* ? */
		nValue = SQL_SP_BETWEEN |
			SQL_SP_COMPARISON |
			SQL_SP_EXISTS |
			SQL_SP_IN |
			SQL_SP_ISNOTNULL |
			SQL_SP_ISNULL |
			SQL_SP_LIKE |
			SQL_SP_MATCH_FULL |
			SQL_SP_MATCH_PARTIAL|
			SQL_SP_MATCH_UNIQUE_FULL |
			SQL_SP_MATCH_UNIQUE_PARTIAL |
			SQL_SP_OVERLAPS |
			SQL_SP_QUANTIFIED_COMPARISON |
			SQL_SP_UNIQUE;
		break;
	case SQL_SQL92_RELATIONAL_JOIN_OPERATORS:
		nValue = SQL_SRJO_CORRESPONDING_CLAUSE |
			SQL_SRJO_CROSS_JOIN |
			SQL_SRJO_EXCEPT_JOIN |
			SQL_SRJO_FULL_OUTER_JOIN |
			SQL_SRJO_INNER_JOIN |
			SQL_SRJO_INTERSECT_JOIN |
			SQL_SRJO_LEFT_OUTER_JOIN |
			SQL_SRJO_NATURAL_JOIN |
			SQL_SRJO_RIGHT_OUTER_JOIN |
			SQL_SRJO_UNION_JOIN;
		break;
	case SQL_SQL92_REVOKE:
		nValue = SQL_SR_CASCADE |
			SQL_SR_DELETE_TABLE |
			SQL_SR_GRANT_OPTION_FOR |
			SQL_SR_INSERT_COLUMN |
			SQL_SR_INSERT_TABLE |
			SQL_SR_REFERENCES_COLUMN |
			SQL_SR_REFERENCES_TABLE |
			SQL_SR_RESTRICT |
			SQL_SR_SELECT_TABLE |
			SQL_SR_UPDATE_COLUMN |
			SQL_SR_UPDATE_TABLE;
		/* SQL_SR_USAGE_ON_DOMAIN |
		 * SQL_SR_USAGE_ON_CHARACTER_SET |
		 * SQL_SR_USAGE_ON_COLLATION |
		 * SQL_SR_USAGE_ON_TRANSLATION */
		break;
	case SQL_SQL92_ROW_VALUE_CONSTRUCTOR: /* ? */
		nValue = SQL_SRVC_VALUE_EXPRESSION |
			SQL_SRVC_NULL |
			SQL_SRVC_DEFAULT |
			SQL_SRVC_ROW_SUBQUERY;
		break;
	case SQL_SQL92_STRING_FUNCTIONS: /* ? */
		nValue = SQL_SSF_CONVERT |
			SQL_SSF_LOWER |
			SQL_SSF_UPPER |
			SQL_SSF_SUBSTRING |
			SQL_SSF_TRANSLATE |
			SQL_SSF_TRIM_BOTH |
			SQL_SSF_TRIM_LEADING |
			SQL_SSF_TRIM_TRAILING;
		break;
	case SQL_SQL92_VALUE_EXPRESSIONS:
		nValue = SQL_SVE_CASE |
			SQL_SVE_CAST |
			SQL_SVE_COALESCE |
			SQL_SVE_NULLIF;
		break;
	case SQL_STANDARD_CLI_CONFORMANCE: /* ? */
		nValue = SQL_SCC_XOPEN_CLI_VERSION1 | SQL_SCC_ISO92_CLI;
		break;
	case SQL_STATIC_CURSOR_ATTRIBUTES1:
		nValue = SQL_CA1_ABSOLUTE |
			SQL_CA1_NEXT |
			SQL_CA1_RELATIVE;
		/* SQL_CA1_BOOKMARK |
		 * SQL_CA1_BULK_ADD |
		 * SQL_CA1_BULK_DELETE_BY_BOOKMARK |
		 * SQL_CA1_BULK_FETCH_BY_BOOKMARK |
		 * SQL_CA1_BULK_UPDATE_BY_BOOKMARK |
		 * SQL_CA1_LOCK_EXCLUSIVE |
		 * SQL_CA1_LOCK_NO_CHANGE |
		 * SQL_CA1_LOCK_UNLOCK |
		 * SQL_CA1_POS_DELETE |
		 * SQL_CA1_POSITIONED_DELETE |
		 * SQL_CA1_POSITIONED_UPDATE |
		 * SQL_CA1_POS_POSITION |
		 * SQL_CA1_POS_REFRESH |
		 * SQL_CA1_POS_UPDATE |
		 * SQL_CA1_SELECT_FOR_UPDATE */
		break;
	case SQL_STATIC_CURSOR_ATTRIBUTES2:
		/* SQL_CA2_CRC_APPROXIMATE |
		 * SQL_CA2_CRC_EXACT |
		 * SQL_CA2_LOCK_CONCURRENCY |
		 * SQL_CA2_MAX_ROWS_AFFECTS_ALL |
		 * SQL_CA2_MAX_ROWS_CATALOG |
		 * SQL_CA2_MAX_ROWS_DELETE |
		 * SQL_CA2_MAX_ROWS_INSERT |
		 * SQL_CA2_MAX_ROWS_SELECT |
		 * SQL_CA2_MAX_ROWS_UPDATE |
		 * SQL_CA2_OPT_ROWVER_CONCURRENCY |
		 * SQL_CA2_OPT_VALUES_CONCURRENCY |
		 * SQL_CA2_READ_ONLY_CONCURRENCY |
		 * SQL_CA2_SENSITIVITY_ADDITIONS |
		 * SQL_CA2_SENSITIVITY_DELETIONS |
		 * SQL_CA2_SENSITIVITY_UPDATES |
		 * SQL_CA2_SIMULATE_NON_UNIQUE |
		 * SQL_CA2_SIMULATE_TRY_UNIQUE |
		 * SQL_CA2_SIMULATE_UNIQUE */
		break;
	case SQL_STRING_FUNCTIONS:
		nValue = SQL_FN_STR_ASCII |
			SQL_FN_STR_BIT_LENGTH |
			SQL_FN_STR_CHAR |
			SQL_FN_STR_CHARACTER_LENGTH |
			SQL_FN_STR_CHAR_LENGTH |
			SQL_FN_STR_CONCAT |
			SQL_FN_STR_DIFFERENCE |
			SQL_FN_STR_INSERT |
			SQL_FN_STR_LCASE |
			SQL_FN_STR_LEFT |
			SQL_FN_STR_LENGTH |
			SQL_FN_STR_LOCATE |
			SQL_FN_STR_LOCATE_2 |
			SQL_FN_STR_LTRIM |
			SQL_FN_STR_OCTET_LENGTH |
			SQL_FN_STR_POSITION |
			SQL_FN_STR_REPEAT |
			SQL_FN_STR_REPLACE |
			SQL_FN_STR_RIGHT |
			SQL_FN_STR_RTRIM |
			SQL_FN_STR_SOUNDEX |
			SQL_FN_STR_SPACE |
			SQL_FN_STR_SUBSTRING |
			SQL_FN_STR_UCASE;
		break;
	case SQL_SUBQUERIES:
		nValue = SQL_SQ_CORRELATED_SUBQUERIES |
			SQL_SQ_COMPARISON |
			SQL_SQ_EXISTS |
			SQL_SQ_IN |
			SQL_SQ_QUANTIFIED;
		break;
	case SQL_SYSTEM_FUNCTIONS:
		nValue = SQL_FN_SYS_DBNAME |
			SQL_FN_SYS_IFNULL |
			SQL_FN_SYS_USERNAME;
		break;
	case SQL_TABLE_TERM:
		sValue = "table";
		break;
	case SQL_TIMEDATE_ADD_INTERVALS:
	case SQL_TIMEDATE_DIFF_INTERVALS:
		/* SQL_FN_TSI_FRAC_SECOND |
		 * SQL_FN_TSI_SECOND |
		 * SQL_FN_TSI_MINUTE |
		 * SQL_FN_TSI_HOUR |
		 * SQL_FN_TSI_DAY |
		 * SQL_FN_TSI_WEEK |
		 * SQL_FN_TSI_MONTH |
		 * SQL_FN_TSI_QUARTER |
		 * SQL_FN_TSI_YEAR */
		break;
	case SQL_TIMEDATE_FUNCTIONS:
		nValue = SQL_FN_TD_CURRENT_DATE |
			SQL_FN_TD_CURRENT_TIME |
			SQL_FN_TD_CURRENT_TIMESTAMP |
			SQL_FN_TD_CURDATE |
			SQL_FN_TD_CURTIME |
			/* SQL_FN_TD_DAYNAME | */
			SQL_FN_TD_DAYOFMONTH |
			SQL_FN_TD_DAYOFWEEK |
			SQL_FN_TD_DAYOFYEAR |
			SQL_FN_TD_EXTRACT |
			SQL_FN_TD_HOUR |
			SQL_FN_TD_MINUTE |
			SQL_FN_TD_MONTH |
			/* SQL_FN_TD_MONTHNAME | */
			SQL_FN_TD_NOW |
			SQL_FN_TD_QUARTER |
			SQL_FN_TD_SECOND |
			/* SQL_FN_TD_TIMESTAMPADD | */
			/* SQL_FN_TD_TIMESTAMPDIFF | */
			SQL_FN_TD_WEEK |
			SQL_FN_TD_YEAR;
		break;
	case SQL_TXN_CAPABLE:
		nValue = SQL_TC_ALL;
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_TXN_ISOLATION_OPTION:
		nValue = SQL_TXN_SERIALIZABLE;
		break;
	case SQL_UNION:
		nValue = SQL_U_UNION | SQL_U_UNION_ALL;
		break;
	case SQL_USER_NAME:
		sValue = dbc->uid ? dbc->uid : "";
		break;
	case SQL_XOPEN_CLI_YEAR: /* ? */
		sValue = "";
		break;

	/* deprecated info types */
	case SQL_FETCH_DIRECTION:
		nValue = SQL_FD_FETCH_ABSOLUTE |
			SQL_FD_FETCH_FIRST |
			SQL_FD_FETCH_LAST |
			SQL_FD_FETCH_NEXT |
			SQL_FD_FETCH_PRIOR |
			SQL_FD_FETCH_RELATIVE;
		/* SQL_FD_FETCH_BOOKMARK */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_LOCK_TYPES:
		nValue = SQL_LCK_NO_CHANGE;
		/*  SQL_LCK_EXCLUSIVE | SQL_LCK_UNLOCK */
		break;
	case SQL_ODBC_API_CONFORMANCE:
		nValue = SQL_OAC_LEVEL2;
		/* SQL_OAC_CORE, SQL_OAC_LEVEL1 */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_ODBC_SQL_CONFORMANCE:
		nValue = SQL_OSC_CORE;
		/* SQL_OSC_MINIMUM, SQL_OSC_EXTENDED */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_POS_OPERATIONS:
		nValue = SQL_POS_POSITION;
		/* SQL_POS_ADD |
		 * SQL_POS_DELETE |
		 * SQL_POS_REFRESH |
		 * SQL_POS_UPDATE */
		break;
	case SQL_POSITIONED_STATEMENTS:
		nValue = SQL_PS_SELECT_FOR_UPDATE;
		/* SQL_PS_POSITIONED_DELETE |
		 * SQL_PS_POSITIONED_UPDATE |
		 * SQL_PS_SELECT_FOR_UPDATE */
		break;
	case SQL_SCROLL_CONCURRENCY:
		nValue = SQL_SCCO_READ_ONLY;
		/* SQL_SCCO_LOCK |
		 * SQL_SCCO_OPT_ROWVER |
		 * SQL_SCCO_OPT_VALUES */
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_STATIC_SENSITIVITY:
		/* SQL_SS_ADDITIONS |
		 * SQL_SS_DELETIONS |
		 * SQL_SS_UPDATES */
		len = sizeof(SQLINTEGER);
		break;

	case SQL_ODBC_SAG_CLI_CONFORMANCE:
		nValue = SQL_OSCC_COMPLIANT;
		len = sizeof(SQLUSMALLINT);
		break;
	case SQL_OUTER_JOINS:
		sValue = "Y";	/* "N" */
		break;

	default:
		/* Invalid information type */
		addDbcError(dbc, "HY096", NULL, 0);
		return SQL_ERROR;
	}

	/* copy the data to the supplied output parameters */
	if (sValue) {
		copyString(sValue, strlen(sValue), InfoValuePtr, BufferLength, StringLengthPtr, SQLSMALLINT, addDbcError, dbc, return SQL_ERROR);
	} else if (InfoValuePtr) {
		if (len == sizeof(SQLULEN))
			*(SQLULEN *) InfoValuePtr = (SQLULEN) nValue;
		else if (len == sizeof(SQLUINTEGER))
			*(SQLUINTEGER *) InfoValuePtr = (SQLUINTEGER) nValue;
		else if (len == sizeof(SQLUSMALLINT))
			*(SQLUSMALLINT *) InfoValuePtr = (SQLUSMALLINT) nValue;
		if (StringLengthPtr)
			*StringLengthPtr = len;
	}

	return dbc->Error ? SQL_SUCCESS_WITH_INFO : SQL_SUCCESS;
}

#ifdef ODBCDEBUG
static char *
translateInfoType(SQLUSMALLINT InfoType)
{
	switch (InfoType) {
	case SQL_ACCESSIBLE_PROCEDURES:
		return "SQL_ACCESSIBLE_PROCEDURES";
	case SQL_ACCESSIBLE_TABLES:
		return "SQL_ACCESSIBLE_TABLES";
	case SQL_ACTIVE_ENVIRONMENTS:
		return "SQL_ACTIVE_ENVIRONMENTS";
	case SQL_AGGREGATE_FUNCTIONS:
		return "SQL_AGGREGATE_FUNCTIONS";
	case SQL_ALTER_DOMAIN:
		return "SQL_ALTER_DOMAIN";
#ifdef SQL_ALTER_SCHEMA
	case SQL_ALTER_SCHEMA:
		return "SQL_ALTER_SCHEMA";
#endif
	case SQL_ALTER_TABLE:
		return "SQL_ALTER_TABLE";
#ifdef SQL_ANSI_SQL_DATETIME_LITERALS
	case SQL_ANSI_SQL_DATETIME_LITERALS:
		return "SQL_ANSI_SQL_DATETIME_LITERALS";
#endif
#ifdef SQL_ASYNC_DBC_FUNCTIONS
	case SQL_ASYNC_DBC_FUNCTIONS:
		return "SQL_ASYNC_DBC_FUNCTIONS";
#endif
	case SQL_ASYNC_MODE:
		return "SQL_ASYNC_MODE";
#ifdef SQL_ASYNC_NOTIFICATION
	case SQL_ASYNC_NOTIFICATION:
		return "SQL_ASYNC_NOTIFICATION";
#endif
	case SQL_BATCH_ROW_COUNT:
		return "SQL_BATCH_ROW_COUNT";
	case SQL_BATCH_SUPPORT:
		return "SQL_BATCH_SUPPORT";
	case SQL_BOOKMARK_PERSISTENCE:
		return "SQL_BOOKMARK_PERSISTENCE";
	case SQL_CATALOG_LOCATION:
		return "SQL_CATALOG_LOCATION";
	case SQL_CATALOG_NAME:
		return "SQL_CATALOG_NAME";
	case SQL_CATALOG_NAME_SEPARATOR:
		return "SQL_CATALOG_NAME_SEPARATOR";
	case SQL_CATALOG_TERM:
		return "SQL_CATALOG_TERM";
	case SQL_CATALOG_USAGE:
		return "SQL_CATALOG_USAGE";
	case SQL_COLLATION_SEQ:
		return "SQL_COLLATION_SEQ";
	case SQL_COLUMN_ALIAS:
		return "SQL_COLUMN_ALIAS";
	case SQL_CONCAT_NULL_BEHAVIOR:
		return "SQL_CONCAT_NULL_BEHAVIOR";
	case SQL_CONVERT_BIGINT:
		return "SQL_CONVERT_BIGINT";
	case SQL_CONVERT_BINARY:
		return "SQL_CONVERT_BINARY";
	case SQL_CONVERT_BIT:
		return "SQL_CONVERT_BIT";
	case SQL_CONVERT_CHAR:
		return "SQL_CONVERT_CHAR";
	case SQL_CONVERT_DATE:
		return "SQL_CONVERT_DATE";
	case SQL_CONVERT_DECIMAL:
		return "SQL_CONVERT_DECIMAL";
	case SQL_CONVERT_DOUBLE:
		return "SQL_CONVERT_DOUBLE";
	case SQL_CONVERT_FLOAT:
		return "SQL_CONVERT_FLOAT";
	case SQL_CONVERT_FUNCTIONS:
		return "SQL_CONVERT_FUNCTIONS";
	case SQL_CONVERT_INTEGER:
		return "SQL_CONVERT_INTEGER";
	case SQL_CONVERT_INTERVAL_DAY_TIME:
		return "SQL_CONVERT_INTERVAL_DAY_TIME";
	case SQL_CONVERT_INTERVAL_YEAR_MONTH:
		return "SQL_CONVERT_INTERVAL_YEAR_MONTH";
	case SQL_CONVERT_LONGVARBINARY:
		return "SQL_CONVERT_LONGVARBINARY";
	case SQL_CONVERT_LONGVARCHAR:
		return "SQL_CONVERT_LONGVARCHAR";
	case SQL_CONVERT_NUMERIC:
		return "SQL_CONVERT_NUMERIC";
	case SQL_CONVERT_REAL:
		return "SQL_CONVERT_REAL";
	case SQL_CONVERT_SMALLINT:
		return "SQL_CONVERT_SMALLINT";
	case SQL_CONVERT_TIME:
		return "SQL_CONVERT_TIME";
	case SQL_CONVERT_TIMESTAMP:
		return "SQL_CONVERT_TIMESTAMP";
	case SQL_CONVERT_TINYINT:
		return "SQL_CONVERT_TINYINT";
	case SQL_CONVERT_VARBINARY:
		return "SQL_CONVERT_VARBINARY";
	case SQL_CONVERT_VARCHAR:
		return "SQL_CONVERT_VARCHAR";
	case SQL_CORRELATION_NAME:
		return "SQL_CORRELATION_NAME";
	case SQL_CREATE_ASSERTION:
		return "SQL_CREATE_ASSERTION";
	case SQL_CREATE_CHARACTER_SET:
		return "SQL_CREATE_CHARACTER_SET";
	case SQL_CREATE_COLLATION:
		return "SQL_CREATE_COLLATION";
	case SQL_CREATE_DOMAIN:
		return "SQL_CREATE_DOMAIN";
	case SQL_CREATE_SCHEMA:
		return "SQL_CREATE_SCHEMA";
	case SQL_CREATE_TABLE:
		return "SQL_CREATE_TABLE";
	case SQL_CREATE_TRANSLATION:
		return "SQL_CREATE_TRANSLATION";
	case SQL_CURSOR_COMMIT_BEHAVIOR:
		return "SQL_CURSOR_COMMIT_BEHAVIOR";
	case SQL_CURSOR_ROLLBACK_BEHAVIOR:
		return "SQL_CURSOR_ROLLBACK_BEHAVIOR";
	case SQL_CURSOR_SENSITIVITY:
		return "SQL_CURSOR_SENSITIVITY";
	case SQL_DATABASE_NAME:
		return "SQL_DATABASE_NAME";
	case SQL_DATA_SOURCE_NAME:
		return "SQL_DATA_SOURCE_NAME";
	case SQL_DATA_SOURCE_READ_ONLY:
		return "SQL_DATA_SOURCE_READ_ONLY";
	case SQL_DBMS_NAME:
		return "SQL_DBMS_NAME";
	case SQL_DBMS_VER:
		return "SQL_DBMS_VER";
	case SQL_DDL_INDEX:
		return "SQL_DDL_INDEX";
	case SQL_DEFAULT_TXN_ISOLATION:
		return "SQL_DEFAULT_TXN_ISOLATION";
	case SQL_DESCRIBE_PARAMETER:
		return "SQL_DESCRIBE_PARAMETER";
	case SQL_DM_VER:
		return "SQL_DM_VER";
#ifdef SQL_DRIVER_AWARE_POOLING_SUPPORTED
	case SQL_DRIVER_AWARE_POOLING_SUPPORTED:
		return "SQL_DRIVER_AWARE_POOLING_SUPPORTED";
#endif
	case SQL_DRIVER_HDBC:
		return "SQL_DRIVER_HDBC";
	case SQL_DRIVER_HDESC:
		return "SQL_DRIVER_HDESC";
	case SQL_DRIVER_HENV:
		return "SQL_DRIVER_HENV";
	case SQL_DRIVER_HLIB:
		return "SQL_DRIVER_HLIB";
	case SQL_DRIVER_HSTMT:
		return "SQL_DRIVER_HSTMT";
	case SQL_DRIVER_NAME:
		return "SQL_DRIVER_NAME";
	case SQL_DRIVER_ODBC_VER:
		return "SQL_DRIVER_ODBC_VER";
	case SQL_DRIVER_VER:
		return "SQL_DRIVER_VER";
	case SQL_DROP_ASSERTION:
		return "SQL_DROP_ASSERTION";
	case SQL_DROP_CHARACTER_SET:
		return "SQL_DROP_CHARACTER_SET";
	case SQL_DROP_COLLATION:
		return "SQL_DROP_COLLATION";
	case SQL_DROP_DOMAIN:
		return "SQL_DROP_DOMAIN";
	case SQL_DROP_SCHEMA:
		return "SQL_DROP_SCHEMA";
	case SQL_DROP_TABLE:
		return "SQL_DROP_TABLE";
	case SQL_DROP_TRANSLATION:
		return "SQL_DROP_TRANSLATION";
	case SQL_DROP_VIEW:
		return "SQL_DROP_VIEW";
	case SQL_DYNAMIC_CURSOR_ATTRIBUTES1:
		return "SQL_DYNAMIC_CURSOR_ATTRIBUTES1";
	case SQL_DYNAMIC_CURSOR_ATTRIBUTES2:
		return "SQL_DYNAMIC_CURSOR_ATTRIBUTES2";
	case SQL_EXPRESSIONS_IN_ORDERBY:
		return "SQL_EXPRESSIONS_IN_ORDERBY";
	case SQL_FILE_USAGE:
		return "SQL_FILE_USAGE";
	case SQL_FORWARD_ONLY_CURSOR_ATTRIBUTES1:
		return "SQL_FORWARD_ONLY_CURSOR_ATTRIBUTES1";
	case SQL_FORWARD_ONLY_CURSOR_ATTRIBUTES2:
		return "SQL_FORWARD_ONLY_CURSOR_ATTRIBUTES2";
	case SQL_GETDATA_EXTENSIONS:
		return "SQL_GETDATA_EXTENSIONS";
	case SQL_GROUP_BY:
		return "SQL_GROUP_BY";
	case SQL_IDENTIFIER_CASE:
		return "SQL_IDENTIFIER_CASE";
	case SQL_IDENTIFIER_QUOTE_CHAR:
		return "SQL_IDENTIFIER_QUOTE_CHAR";
	case SQL_INDEX_KEYWORDS:
		return "SQL_INDEX_KEYWORDS";
	case SQL_INFO_SCHEMA_VIEWS:
		return "SQL_INFO_SCHEMA_VIEWS";
	case SQL_INSERT_STATEMENT:
		return "SQL_INSERT_STATEMENT";
	case SQL_INTEGRITY:
		return "SQL_INTEGRITY";
	case SQL_KEYSET_CURSOR_ATTRIBUTES1:
		return "SQL_KEYSET_CURSOR_ATTRIBUTES1";
	case SQL_KEYSET_CURSOR_ATTRIBUTES2:
		return "SQL_KEYSET_CURSOR_ATTRIBUTES2";
	case SQL_KEYWORDS:
		return "SQL_KEYWORDS";
	case SQL_LIKE_ESCAPE_CLAUSE:
		return "SQL_LIKE_ESCAPE_CLAUSE";
	case SQL_MAX_ASYNC_CONCURRENT_STATEMENTS:
		return "SQL_MAX_ASYNC_CONCURRENT_STATEMENTS";
	case SQL_MAX_BINARY_LITERAL_LEN:
		return "SQL_MAX_BINARY_LITERAL_LEN";
	case SQL_MAX_CATALOG_NAME_LEN:
		return "SQL_MAX_CATALOG_NAME_LEN";
	case SQL_MAX_CHAR_LITERAL_LEN:
		return "SQL_MAX_CHAR_LITERAL_LEN";
	case SQL_MAX_COLUMN_NAME_LEN:
		return "SQL_MAX_COLUMN_NAME_LEN";
	case SQL_MAX_COLUMNS_IN_GROUP_BY:
		return "SQL_MAX_COLUMNS_IN_GROUP_BY";
	case SQL_MAX_COLUMNS_IN_INDEX:
		return "SQL_MAX_COLUMNS_IN_INDEX";
	case SQL_MAX_COLUMNS_IN_ORDER_BY:
		return "SQL_MAX_COLUMNS_IN_ORDER_BY";
	case SQL_MAX_COLUMNS_IN_SELECT:
		return "SQL_MAX_COLUMNS_IN_SELECT";
	case SQL_MAX_COLUMNS_IN_TABLE:
		return "SQL_MAX_COLUMNS_IN_TABLE";
	case SQL_MAX_CONCURRENT_ACTIVITIES:
		return "SQL_MAX_CONCURRENT_ACTIVITIES";
	case SQL_MAX_CURSOR_NAME_LEN:
		return "SQL_MAX_CURSOR_NAME_LEN";
	case SQL_MAX_DRIVER_CONNECTIONS:
		return "SQL_MAX_DRIVER_CONNECTIONS";
	case SQL_MAX_IDENTIFIER_LEN:
		return "SQL_MAX_IDENTIFIER_LEN";
	case SQL_MAX_INDEX_SIZE:
		return "SQL_MAX_INDEX_SIZE";
	case SQL_MAX_PROCEDURE_NAME_LEN:
		return "SQL_MAX_PROCEDURE_NAME_LEN";
	case SQL_MAX_ROW_SIZE:
		return "SQL_MAX_ROW_SIZE";
	case SQL_MAX_ROW_SIZE_INCLUDES_LONG:
		return "SQL_MAX_ROW_SIZE_INCLUDES_LONG";
	case SQL_MAX_SCHEMA_NAME_LEN:
		return "SQL_MAX_SCHEMA_NAME_LEN";
	case SQL_MAX_STATEMENT_LEN:
		return "SQL_MAX_STATEMENT_LEN";
	case SQL_MAX_TABLE_NAME_LEN:
		return "SQL_MAX_TABLE_NAME_LEN";
	case SQL_MAX_TABLES_IN_SELECT:
		return "SQL_MAX_TABLES_IN_SELECT";
	case SQL_MAX_USER_NAME_LEN:
		return "SQL_MAX_USER_NAME_LEN";
	case SQL_MULTIPLE_ACTIVE_TXN:
		return "SQL_MULTIPLE_ACTIVE_TXN";
	case SQL_MULT_RESULT_SETS:
		return "SQL_MULT_RESULT_SETS";
	case SQL_NEED_LONG_DATA_LEN:
		return "SQL_NEED_LONG_DATA_LEN";
	case SQL_NON_NULLABLE_COLUMNS:
		return "SQL_NON_NULLABLE_COLUMNS";
	case SQL_NULL_COLLATION:
		return "SQL_NULL_COLLATION";
	case SQL_NUMERIC_FUNCTIONS:
		return "SQL_NUMERIC_FUNCTIONS";
	case SQL_ODBC_INTERFACE_CONFORMANCE:
		return "SQL_ODBC_INTERFACE_CONFORMANCE";
#ifdef SQL_ODBC_STANDARD_CLI_CONFORMANCE
	case SQL_ODBC_STANDARD_CLI_CONFORMANCE:
		return "SQL_ODBC_STANDARD_CLI_CONFORMANCE";
#endif
	case SQL_ODBC_VER:
		return "SQL_ODBC_VER";
	case SQL_OJ_CAPABILITIES:
		return "SQL_OJ_CAPABILITIES";
	case SQL_ORDER_BY_COLUMNS_IN_SELECT:
		return "SQL_ORDER_BY_COLUMNS_IN_SELECT";
	case SQL_OUTER_JOINS:
		return "SQL_OUTER_JOINS";
	case SQL_PARAM_ARRAY_ROW_COUNTS:
		return "SQL_PARAM_ARRAY_ROW_COUNTS";
	case SQL_PARAM_ARRAY_SELECTS:
		return "SQL_PARAM_ARRAY_SELECTS";
	case SQL_PROCEDURES:
		return "SQL_PROCEDURES";
	case SQL_PROCEDURE_TERM:
		return "SQL_PROCEDURE_TERM";
	case SQL_QUOTED_IDENTIFIER_CASE:
		return "SQL_QUOTED_IDENTIFIER_CASE";
	case SQL_ROW_UPDATES:
		return "SQL_ROW_UPDATES";
	case SQL_SCHEMA_TERM:
		return "SQL_SCHEMA_TERM";
	case SQL_SCHEMA_USAGE:
		return "SQL_SCHEMA_USAGE";
	case SQL_SCROLL_OPTIONS:
		return "SQL_SCROLL_OPTIONS";
	case SQL_SEARCH_PATTERN_ESCAPE:
		return "SQL_SEARCH_PATTERN_ESCAPE";
	case SQL_SERVER_NAME:
		return "SQL_SERVER_NAME";
	case SQL_SPECIAL_CHARACTERS:
		return "SQL_SPECIAL_CHARACTERS";
	case SQL_SQL_CONFORMANCE:
		return "SQL_SQL_CONFORMANCE";
	case SQL_STATIC_CURSOR_ATTRIBUTES1:
		return "SQL_STATIC_CURSOR_ATTRIBUTES1";
	case SQL_STATIC_CURSOR_ATTRIBUTES2:
		return "SQL_STATIC_CURSOR_ATTRIBUTES2";
	case SQL_STRING_FUNCTIONS:
		return "SQL_STRING_FUNCTIONS";
	case SQL_SUBQUERIES:
		return "SQL_SUBQUERIES";
	case SQL_SYSTEM_FUNCTIONS:
		return "SQL_SYSTEM_FUNCTIONS";
	case SQL_TABLE_TERM:
		return "SQL_TABLE_TERM";
	case SQL_TIMEDATE_ADD_INTERVALS:
		return "SQL_TIMEDATE_ADD_INTERVALS";
	case SQL_TIMEDATE_DIFF_INTERVALS:
		return "SQL_TIMEDATE_DIFF_INTERVALS";
	case SQL_TIMEDATE_FUNCTIONS:
		return "SQL_TIMEDATE_FUNCTIONS";
	case SQL_TXN_CAPABLE:
		return "SQL_TXN_CAPABLE";
	case SQL_TXN_ISOLATION_OPTION:
		return "SQL_TXN_ISOLATION_OPTION";
	case SQL_UNION:
		return "SQL_UNION";
	case SQL_USER_NAME:
		return "SQL_USER_NAME";
	case SQL_XOPEN_CLI_YEAR:
		return "SQL_XOPEN_CLI_YEAR";
	default:
		return "unknown";
	}
}
#endif

SQLRETURN SQL_API
SQLGetInfo(SQLHDBC ConnectionHandle,
	   SQLUSMALLINT InfoType,
	   SQLPOINTER InfoValuePtr,
	   SQLSMALLINT BufferLength,
	   SQLSMALLINT *StringLengthPtr)
{
	ODBCDbc *dbc = (ODBCDbc *) ConnectionHandle;

#ifdef ODBCDEBUG
	ODBCLOG("SQLGetInfo " PTRFMT " %s\n",
		PTRFMTCAST ConnectionHandle, translateInfoType(InfoType));
#endif

	if (!isValidDbc(dbc))
		return SQL_INVALID_HANDLE;

	clearDbcErrors(dbc);

	return SQLGetInfo_(dbc,
			   InfoType,
			   InfoValuePtr,
			   BufferLength,
			   StringLengthPtr);
}

SQLRETURN SQL_API
SQLGetInfoA(SQLHDBC ConnectionHandle,
	    SQLUSMALLINT InfoType,
	    SQLPOINTER InfoValuePtr,
	    SQLSMALLINT BufferLength,
	    SQLSMALLINT *StringLengthPtr)
{
	return SQLGetInfo(ConnectionHandle,
			  InfoType,
			  InfoValuePtr,
			  BufferLength,
			  StringLengthPtr);
}

SQLRETURN SQL_API
SQLGetInfoW(SQLHDBC ConnectionHandle,
	    SQLUSMALLINT InfoType,
	    SQLPOINTER InfoValuePtr,
	    SQLSMALLINT BufferLength,
	    SQLSMALLINT *StringLengthPtr)
{
	ODBCDbc *dbc = (ODBCDbc *) ConnectionHandle;
	SQLRETURN rc;
	SQLPOINTER ptr;
	SQLSMALLINT n;

#ifdef ODBCDEBUG
	ODBCLOG("SQLGetInfoW " PTRFMT " %s\n",
		PTRFMTCAST ConnectionHandle, translateInfoType(InfoType));
#endif

	if (!isValidDbc(dbc))
		return SQL_INVALID_HANDLE;

	clearDbcErrors(dbc);

	switch (InfoType) {
	/* all string attributes */
	case SQL_ACCESSIBLE_PROCEDURES:
	case SQL_ACCESSIBLE_TABLES:
	case SQL_CATALOG_NAME:
	case SQL_CATALOG_NAME_SEPARATOR:
	case SQL_CATALOG_TERM:
	case SQL_COLLATION_SEQ:
	case SQL_COLUMN_ALIAS:
	case SQL_DATABASE_NAME:
	case SQL_DATA_SOURCE_NAME:
	case SQL_DATA_SOURCE_READ_ONLY:
	case SQL_DBMS_NAME:
	case SQL_DBMS_VER:
	case SQL_DESCRIBE_PARAMETER:
	case SQL_DM_VER:
	case SQL_DRIVER_NAME:
	case SQL_DRIVER_ODBC_VER:
	case SQL_DRIVER_VER:
	case SQL_EXPRESSIONS_IN_ORDERBY:
	case SQL_IDENTIFIER_QUOTE_CHAR:
	case SQL_INTEGRITY:
	case SQL_KEYWORDS:
	case SQL_LIKE_ESCAPE_CLAUSE:
	case SQL_MAX_ROW_SIZE_INCLUDES_LONG:
	case SQL_MULTIPLE_ACTIVE_TXN:
	case SQL_MULT_RESULT_SETS:
	case SQL_NEED_LONG_DATA_LEN:
	case SQL_ODBC_VER:
	case SQL_ORDER_BY_COLUMNS_IN_SELECT:
	case SQL_OUTER_JOINS:
	case SQL_PROCEDURES:
	case SQL_PROCEDURE_TERM:
	case SQL_ROW_UPDATES:
	case SQL_SCHEMA_TERM:
	case SQL_SEARCH_PATTERN_ESCAPE:
	case SQL_SERVER_NAME:
	case SQL_SPECIAL_CHARACTERS:
	case SQL_TABLE_TERM:
	case SQL_USER_NAME:
	case SQL_XOPEN_CLI_YEAR:
		rc = SQLGetInfo_(dbc, InfoType, NULL, 0, &n);
		if (!SQL_SUCCEEDED(rc))
			return rc;
		clearDbcErrors(dbc);
		n++;		/* account for NUL byte */
		ptr = (SQLPOINTER) malloc(n);
		if (ptr == NULL) {
			/* Memory allocation error */
			addDbcError(dbc, "HY001", NULL, 0);
			return SQL_ERROR;
		}
		break;
	default:
		n = BufferLength;
		ptr = InfoValuePtr;
		break;
	}

	rc = SQLGetInfo_(dbc, InfoType, ptr, n, &n);

	if (ptr != InfoValuePtr) {
		if (SQL_SUCCEEDED(rc)) {
			fixWcharOut(rc, ptr, n, InfoValuePtr, BufferLength,
				    StringLengthPtr, 2, addDbcError, dbc);
		}
		free(ptr);
	} else if (StringLengthPtr)
		*StringLengthPtr = n;

	return rc;
}
