-- Convert schema 'kddart_v2_4_0_mysql.sql' to 'kddart_v2_5_0_mysql.sql':;

BEGIN;

ALTER TABLE designtype CHANGE COLUMN DesignTemplateFile DesignTemplateFile text NULL comment 'The template file defines how the parameter are required to be inserted in the input file for the design software.',
                       ENGINE=InnoDB  DEFAULT CHARACTER SET utf8 comment='Each trial (experiment) has a design (layout/method) describing how the genotypes have been planted, the number of replicates, etc. Design type identifies the type of a trial design. For example: - Randomised Block using DiGGer - The trial design of this design type is generated by the DiGGer. - Unknown Design Type. ';

ALTER TABLE deviceregister ADD COLUMN DeviceConf text NULL comment 'Device configuration store - can be object e.g. json or yaml string etc',
                           ENGINE=InnoDB  DEFAULT CHARACTER SET utf8 comment='Register of devices, which have access to the database to supply data. Devices to include those that supply environmental data or seed inventory data (e.g. scales) and extend to those that supply for phenotypic measurements.  One device can record more than one data type / parameter to more than one environment layer.';

ALTER TABLE generaltype CHANGE COLUMN Class Class set('site', 'item', 'container', 'deviceregister', 'trial', 'trialevent', 'sample', 'specimengroup', 'specimengroupstatus', 'state', 'parent', 'itemparent', 'genotypespecimen', 'markerdataset', 'workflow', 'project', 'itemlog', 'plate', 'genmap', 'multimedia', 'tissue', 'genotypealias', 'genparent', 'genotypealiasstatus', 'traitgroup', 'unittype', 'trialgroup', 'breedingmethod', 'traitdatatype', 'season') NOT NULL comment 'class of type - possible values (site, item, container, deviceregister, trial, operation, sample, specimengroup, specimengroupstatus, state, parent, itemparent, genotypespecimen, markerdataset, workflow, project, itemlog, plate, genmap, multimedia, tissue, genotypealias, genparent, genotypealiasstatus, traitgroup, unittype, trialgroup, breedingmethod, traitdatatype, season)',
                        ENGINE=InnoDB  DEFAULT CHARACTER SET utf8 comment='Contains global vocabularies of types to support classification. The class column lists the possible classes of types.  It also spans to types in Marker Module. With one exception of designtype table for trial all columns named *type refer to particular class of the types in this table. There are also some *status classifications contained here. Classes are: site, item, container, deviceregister, trial, operation, sample, specimengroup, specimengroupstatus, state, parent, itemparent, genotypespecimen, markerdataset, workflow, project, itemlog, plate, genmap, multimedia, tissue, genotypealias, genparent, genotypealiasstatus, traitgroup, unittype, trialgroup, breedingmethod, traitdatatype, season';

ALTER TABLE trait ADD COLUMN AltIdentifier varchar(254) NULL DEFAULT NULL comment 'alternative identifier e.g. code used in another system like trait ontology or other',
                  ADD UNIQUE xt_AltIdentifier (AltIdentifier);

ALTER TABLE traitgroup ADD COLUMN AltIdentifier varchar(254) NULL DEFAULT NULL comment 'alternative identified e.g. code used in another system like trait ontology',
                       ADD UNIQUE trg_AltIdentifier (AltIdentifier),
                       ENGINE=InnoDB  DEFAULT CHARACTER SET utf8 comment='Group of traits. Can be used for various arbitrary purposes.';

ALTER TABLE trial ADD COLUMN SeasonId integer(11) NULL comment 'id of the season type from general type - to identify global concept of seasons',
                  ADD COLUMN TrialLayout text NULL comment 'object describing trial units layout - e.g. walking path, relative location of raw1, col1, etc',
                  ADD COLUMN TULastUpdateTimeStamp datetime NOT NULL comment 'date and time of trial unit last update - used for bulk update of trial units',
                  ADD INDEX xtr_SeasonId (SeasonId),
                  ENGINE=InnoDB  DEFAULT CHARACTER SET utf8 comment='This table contains list of trials (equivalent to field experiments) performed within sites. ';


ALTER TABLE trialgroup ADD COLUMN TrialGroupLayout text NULL comment 'layout of the trials in the group, e.g. relative positions to each other';

ALTER TABLE trialunitspecimen ADD COLUMN SpecimenNumber integer(11) NOT NULL DEFAULT 0 comment 'distinguish between instances of the same specimen in the same trial unit - just number them',
                              ADD COLUMN TUSLabel varchar(254) NULL DEFAULT NULL comment 'label for trial unit specimen - anything to make label on individual',
                              ADD UNIQUE xtus_USN (SpecimenId, TrialUnitId, SpecimenNumber),
                              ENGINE=InnoDB  DEFAULT CHARACTER SET utf8 comment='Maintains the many to many relationship between trialunit and specimens. If specimen is the representation of the whole genotype (or composition of genotypes), it will have a single record here. Allows the sampling of multiple specimens from a unit or even same specimen multiple times in the unit distinguished by specimen number.  For instances of long lived trials (e.g. tree living 30 years) this enables the reassignment of the same plant into different trial units.';

COMMIT;

BEGIN;

ALTER TABLE genotypealias ADD COLUMN IsGenotypeName tinyint(4) NOT NULL DEFAULT 0 comment 'flag if this name is a current genotype name - can point to only single record in all genotype records',
                          ADD COLUMN GenusId integer(11) NULL comment 'genus id - same as for main genotype record - to assure uniqness inside the genus - should be managed by trigger',
                          ENGINE=InnoDB  DEFAULT CHARACTER SET utf8 comment='One genotype may have many historical names (aliases) under which it has been known.';

COMMIT;

BEGIN;

UPDATE genotypealias LEFT JOIN genotype ON genotypealias.GenotypeId = genotype.GenotypeId SET genotypealias.GenusId = genotype.GenusId;

CREATE TEMPORARY TABLE IF NOT EXISTS uniq_genotypealias AS (SELECT MIN(GenotypeAliasId) AS KeepGenotypeAliasId FROM genotypealias GROUP BY GenusId, GenotypeAliasName);

CREATE TEMPORARY TABLE IF NOT EXISTS del_genotypealias AS (SELECT GenotypeAliasId FROM genotypealias WHERE GenotypeAliasId NOT IN (SELECT KeepGenotypeAliasId FROM uniq_genotypealias));

DELETE FROM a USING genotypealias AS a INNER JOIN del_genotypealias AS b ON a.GenotypeAliasId = b.GenotypeAliasId;

DROP TABLE uniq_genotypealias;

DROP TABLE del_genotypealias;


COMMIT;


-- triggers for specimen permissions update, unique genotype and alias naming

DELIMITER //

-- This is to do a combined uniqueness checking between GenotypeName in genotype table
-- and GenotypeAliasName in genotypealias table. MySQL uniqueness constraint cannot be
-- done across multiple tables. In the genotype table, there is a uniquness constraint
-- for GenotypeName and GenusId. However, because the error interaction between table
-- uniqueness constaint with the trigger, this trigger checks GenotypeName, GenusId
-- uniqueness constaint again in addition to the GenotypeAliasName. If the uniqueness
-- constraints in genotype and genotypealias tables are not met, there will be an error
-- when there is an attempt to add a new genotype record.

DROP TRIGGER IF EXISTS `genotype_before_insert`;
CREATE TRIGGER genotype_before_insert 
BEFORE INSERT ON genotype 
FOR EACH ROW
BEGIN
  DECLARE count_geno INT DEFAULT 0;
  DECLARE count_alias INT DEFAULT 0;
  DECLARE msg VARCHAR(255);

	SET count_geno = (SELECT COUNT(*) FROM genotype WHERE GenotypeName=NEW.GenotypeName AND GenusId=NEW.GenusId);

	IF count_geno > 0 THEN
		SET msg = concat('Trigger genotype_before_insert error: genotype name (', NEW.GenotypeName, ') GenusId (', CAST(NEW.GenusId AS CHAR), ') already exists.');
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = msg;
	END IF;

  SET count_alias = (SELECT COUNT(*) FROM genotypealias WHERE GenotypeAliasName=NEW.GenotypeName AND GenusId=NEW.GenusId);

  IF count_alias > 0 THEN
    SET msg = concat('Trigger genotype_before_insert error: genotype name (', NEW.GenotypeName, ') GenusId (', CAST(NEW.GenusId AS CHAR), ') already exists in genotypealias table.');
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = msg;
  END IF;

END//

-- This trigger is similar to genotype_before_insert trigger that it checks the constraints in 
-- genotype and genotypealias tables for the combined uniqueness among GenotypeName and
-- GenotypeAliasName within the same genus. When these constraints are not met, there will be
-- an error when there is an attempt to update an existing genotype record. 

DROP TRIGGER IF EXISTS `genotype_before_update`;
CREATE TRIGGER genotype_before_update 
BEFORE UPDATE ON genotype 
FOR EACH ROW
BEGIN
  DECLARE count_geno INT DEFAULT 0;
  DECLARE count_alias INT DEFAULT 0;
  DECLARE geno_alias_id INT DEFAULT 0;
  DECLARE msg VARCHAR(255);

  SET geno_alias_id = (SELECT GenotypeAliasId FROM genotypealias WHERE GenotypeId=NEW.GenotypeId AND GenotypeAliasName=NEW.GenotypeName);

  IF geno_alias_id IS NULL THEN
     SET geno_alias_id = 0;
  END IF;

	SET count_geno = (SELECT COUNT(*) FROM genotype WHERE GenotypeName=NEW.GenotypeName AND GenusId=NEW.GenusId AND GenotypeId<>NEW.GenotypeId);

	IF count_geno > 0 THEN
		SET msg = concat('Trigger genotype_before_update error: genotype name (', NEW.GenotypeName, ') GenusId (', CAST(NEW.GenusId AS CHAR), ') already exists.');
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = msg;
	END IF;

  SET count_alias = (SELECT COUNT(*) FROM genotypealias WHERE GenotypeAliasName=NEW.GenotypeName AND GenusId=NEW.GenusId AND GenotypeAliasId<>geno_alias_id);

  IF count_alias > 0 THEN
    SET msg = concat('Trigger genotype_before_update error: genotype name (', NEW.GenotypeName, ') GenusId (', CAST(NEW.GenusId AS CHAR), ') already exists in genotypealias table.');
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = msg;
  END IF;

END//

-- This trigger is for replicating genotype permission information to specimen table in order to
-- reduce the need to join genotype with specimen via genotypespecimen table and for replicating
-- GenotypeName and GenusId from genotype to genotypealias table.

DROP TRIGGER IF EXISTS `genotype_after_update`;
CREATE TRIGGER genotype_after_update 
AFTER UPDATE ON genotype 
FOR EACH ROW
BEGIN
  DECLARE old_geno_alias_id INT DEFAULT 0;
  DECLARE new_geno_alias_id INT DEFAULT 0;

  SET old_geno_alias_id = (SELECT GenotypeAliasId FROM genotypealias WHERE GenotypeId=OLD.GenotypeId AND GenotypeAliasName=OLD.GenotypeName);

  IF old_geno_alias_id IS NOT NULL THEN
     UPDATE genotypealias SET
        IsGenotypeName=0
     WHERE
        GenotypeAliasId=old_geno_alias_id;
  END IF;

  SET new_geno_alias_id = (SELECT GenotypeAliasId FROM genotypealias WHERE GenotypeId=NEW.GenotypeId AND GenotypeAliasName=NEW.GenotypeName);

  IF new_geno_alias_id IS NOT NULL THEN
     UPDATE genotypealias SET
       IsGenotypeName=1
     WHERE
       GenotypeAliasId=new_geno_alias_id;
  ELSE
     INSERT INTO genotypealias SET
       GenotypeId=NEW.GenotypeId,
       GenotypeAliasName=NEW.GenotypeName,
       IsGenotypeName=1,
       GenusId=NEW.GenusId;
  END IF;

  IF ( NEW.OwnGroupId <> OLD.OwnGroupId OR
       NEW.OwnGroupPerm <> OLD.OwnGroupPerm OR
       NEW.AccessGroupId <> OLD.AccessGroupId OR
       NEW.AccessGroupPerm <> OLD.AccessGroupPerm OR
       NEW.OtherPerm <> OLD.OtherPerm ) THEN

    UPDATE
      specimen LEFT JOIN genotypespecimen ON specimen.SpecimenId = genotypespecimen.SpecimenId
    SET
      specimen.OwnGroupId      = NEW.OwnGroupId,
      specimen.AccessGroupId   = NEW.AccessGroupId,
      specimen.OwnGroupPerm    = NEW.OwnGroupPerm,
      specimen.AccessGroupPerm = NEW.AccessGroupPerm,
      specimen.OtherPerm       = NEW.OtherPerm
    WHERE
      genotypespecimen.GenotypeId=NEW.GenotypeId AND genotypespecimen.InheritanceFlag=1;

  END IF;

END//

-- This trigger is for creating GenotypeAlias automatically when a new genotype
-- record is added to the database.

DROP TRIGGER IF EXISTS `genotype_after_insert`;
CREATE TRIGGER genotype_after_insert
AFTER INSERT ON genotype
FOR EACH ROW
BEGIN
  INSERT INTO genotypealias SET
    GenotypeId=NEW.GenotypeId,
    GenotypeAliasName=NEW.GenotypeName,
    IsGenotypeName=1,
    GenusId=NEW.GenusId;
END//

-- This trigger is for cross checking the GenotypeAliasName against GenotypeName
-- in genotype table. It also checks it against GenotypeAliasName in genotypealias
-- table. It is redundant to the unique constraint on GenotypeAliasName, GenusId
-- uniqueness. However, this extra checking is here to make this trigger consistent
-- with other triggers.

DROP TRIGGER IF EXISTS `genotypealias_before_insert`;
CREATE TRIGGER genotypealias_before_insert
BEFORE INSERT ON genotypealias
FOR EACH ROW
BEGIN
  DECLARE count_geno INT DEFAULT 0;
  DECLARE count_alias INT DEFAULT 0;
  DECLARE genus_id INT DEFAULT 0;
  DECLARE msg VARCHAR(255);

  SET genus_id = (SELECT GenusId FROM genotype WHERE GenotypeId=NEW.GenotypeId);

  IF NEW.GenusId <> genus_id THEN
    SET msg = concat('Trigger genotypealias_before_insert error: GenusId (', CAST(NEW.GenusId AS CHAR), ') provided is not consistent with GenusId (', CAST(genus_id AS CHAR), ') in the genotype.');
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = msg;
  END IF;

  SET count_geno = (SELECT COUNT(*) FROM genotype WHERE GenotypeName=NEW.GenotypeAliasName AND GenusId=genus_id AND GenotypeId<>NEW.GenotypeId);

  IF count_geno > 0 THEN
    SET msg = concat('Trigger genotypealias_before_insert error: GenotypeAliasName (', NEW.GenotypeAliasName, ') is already used as GenotypeName in another genotype.');
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = msg;
  END IF;

  SET count_alias = (SELECT COUNT(*) FROM genotypealias WHERE GenotypeAliasName=NEW.GenotypeAliasName AND GenusId=genus_id);

  IF count_alias > 0 THEN
    SET msg = concat('Trigger genotypealias_before_insert error: GenotypeAliasName (', NEW.GenotypeAliasName, ') already exists.');
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = msg;
  END IF;

END//

--

DROP TRIGGER IF EXISTS `genotypealias_before_update`;
CREATE TRIGGER genotypealias_before_update
BEFORE UPDATE ON genotypealias
FOR EACH ROW
BEGIN
  DECLARE count_geno INT DEFAULT 0;
  DECLARE count_alias INT DEFAULT 0;
  DECLARE genus_id INT DEFAULT 0;
  DECLARE msg VARCHAR(255);

  SET genus_id = (SELECT GenusId FROM genotype WHERE GenotypeId=NEW.GenotypeId);

  IF NEW.GenusId <> genus_id THEN
    SET msg = concat('Trigger genotypealias_before_update error: GenusId (', CAST(NEW.GenusId AS CHAR), ') provided is not consistent with GenusId (', CAST(genus_id AS CHAR), ') in the genotype.');
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = msg;
  END IF;

  SET count_geno = (SELECT COUNT(*) FROM genotype WHERE GenotypeName=NEW.GenotypeAliasName AND GenusId=genus_id AND GenotypeId<>NEW.GenotypeId);

  IF count_geno > 0 THEN
    SET msg = concat('Trigger genotypealias_before_update error: GenotypeAliasName (', NEW.GenotypeAliasName, ') is already used as GenotypeName in another genotype.');
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = msg;
  END IF;

  SET count_alias = (SELECT COUNT(*) FROM genotypealias WHERE GenotypeAliasName=NEW.GenotypeAliasName AND GenusId=genus_id AND GenotypeAliasId<>NEW.GenotypeAliasId);

  IF count_alias > 0 THEN
    SET msg = concat('Trigger genotypealias_before_update error: GenotypeAliasName (', NEW.GenotypeAliasName, ') already exists.');
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = msg;
  END IF;

  --
END//

DELIMITER ;

-- This statement is make GenotypeName be added as GenotypeAliasName in genotypelias via the triggers
UPDATE genotype SET GenotypeName=GenotypeName;

ALTER TABLE genotypealias CHANGE COLUMN GenusId GenusId integer(11) NOT NULL comment 'genus id - same as for main genotype record - to assure uniqness inside the genus - should be managed by trigger',
                          ADD UNIQUE xga_GenotypeAliasUniq (GenusId, GenotypeAliasName),
                          ADD CONSTRAINT genotypealias_ibfk_4 FOREIGN KEY (GenusId) REFERENCES genus (GenusId) ON DELETE NO ACTION ON UPDATE NO ACTION,
                          ENGINE=InnoDB  DEFAULT CHARACTER SET utf8 comment='One genotype may have many historical names (aliases) under which it has been known.';


BEGIN;

SELECT TypeId+1 INTO @season_id FROM generaltype ORDER BY TypeId DESC LIMIT 1;

INSERT INTO generaltype(TypeId,Class,TypeName,IsTypeActive,IsFixed) Values(@season_id,'season','UnknownSeasonFromMigration',1,0);

UPDATE trial SET SeasonId=@season_id;

ALTER TABLE trial CHANGE COLUMN SeasonId SeasonId integer(11) NOT NULL comment 'id of the season type from general type - to identify global concept of seasons',
                  ADD CONSTRAINT trial_ibfk_7 FOREIGN KEY (SeasonId) REFERENCES generaltype (TypeId) ON DELETE NO ACTION ON UPDATE NO ACTION,
                  ENGINE=InnoDB  DEFAULT CHARACTER SET utf8 comment='This table contains list of trials (equivalent to field experiments) performed within sites. ';

COMMIT;
